<?php

declare(strict_types=1);

namespace Juweliere\CheckoutBundle\Controller;

use Juweliere\ApiBundle\Service\JuwApi;
use Juweliere\CheckoutBundle\Checkout\Order\Item;
use Juweliere\CheckoutBundle\Checkout\Order\Order;
use Juweliere\CheckoutBundle\Checkout\OrderHandler;
use Juweliere\CheckoutBundle\Checkout\ProductDataProvider;
use Juweliere\CheckoutBundle\Checkout\SessionHandler;
use Juweliere\CheckoutBundle\Checkout\StripeHandler;
use Juweliere\CheckoutBundle\Checkout\TrackingDataProvider;
use Juweliere\CheckoutBundle\Entity\OrderStripe;
use Psr\Log\LoggerInterface;
use Stripe\Charge;
use Stripe\Checkout\Session;
use Stripe\PaymentIntent;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Generator\UrlGenerator;
use Twig\Environment;
use Sulu\Bundle\WebsiteBundle\Controller\WebsiteController;
use Symfony\Component\Routing\Annotation\Route;

/**
 * @Route("/checkout", requirements={"_locale": "en|es|fr"}, name="juweliere_checkout_checkout_")
 */
class CheckoutController extends WebsiteController
{
    const PRODUCT_TYPE_WATCH = "uhren";
    const PRODUCT_TYPE_JEWELLERY = "schmuck";
    const SESSION_LAST_SEEN = "lastseen";

    private SessionHandler $sessionHandler;
    private ProductDataProvider $productDataProvider;
    private bool $isProd = true;

    /**
     * @param SessionHandler $sessionHandler
     * @param ProductDataProvider $productDataProvider
     */
    public function __construct(SessionHandler $sessionHandler, ProductDataProvider $productDataProvider, ParameterBagInterface $appParams) {
        $this->sessionHandler = $sessionHandler;
        $this->productDataProvider = $productDataProvider;
        $this->isProd = $appParams->get('kernel.environment') === 'prod';
    }

    /**
     * @Route("/test123", name="test123", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function test(Request $request, JuwApi $juwApi): Response
    {
        var_dump(json_encode($juwApi->getMe()));
        die();
    }

    /**
     * @Route("/", name="basket", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function basket(Request $request): Response
    {
        $template = '@JuweliereCheckout/pages/checkout/basket.html.twig';

        $response = $this->render(
            $template,
            array (
                "items" => $this->getSessionOrderItems()
            )
        );

        return $response;
    }

    /**
     * @Route("/sku/{sku}", name="sku", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function sku(Request $request, string $sku, SessionHandler $sessionHandler): Response
    {
        $sessionHandler->addItemToOrder($sku);

        return $this->redirect("/checkout");
    }

    /**
     * @Route("/address", name="address", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function address(Request $request): Response
    {
        $template = '@JuweliereCheckout/pages/checkout/address.html.twig';

        $response = $this->render(
            $template,
            array (
                "items" => $this->getSessionOrderItems()
            )
        );

        return $response;
    }

    /**
     * @Route("/stripetest/{sku}", name="stripetest", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function stripetest(Request $request, $sku, ProductDataProvider $productDataProvider, StripeHandler $stripe): Response
    {
        if (!$this->isProd) {
            return $this->redirect("/checkout");
        }

        return $this->stripe($request, $sku, $productDataProvider, $stripe, 111, "!!! TEST !!! ");
    }

    /**
     * @Route("/stripe/{sku}", name="stripe", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function stripe(Request $request, string $sku, ProductDataProvider $productDataProvider, StripeHandler $stripe, $price = null, $prefix = ""): Response
    {
        /** @var string $template */
        $template = '@JuweliereCheckout/pages/checkout/stripe.html.twig';

        /** @var array $product */
        $product = $productDataProvider->getProductBySku($sku);

        if (!is_null($price) && !$this->isProd) {
            $product["price"] =  $price;
        }

        $product["name"] = $prefix . $product["name"];
        $product["shortDescription"] = $prefix . $product["shortDescription"];

        $sessionId = $stripe->createCheckoutSession($product);

        $response = $this->render($template, ["session"=> $sessionId]);

        return $response;
    }

    /**
     * @Route("/payment", name="payment", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function payment(Request $request, SessionHandler $session, StripeHandler $stripe): Response
    {
        $this->sessionHandler->updateUserData($request);

        $template = '@JuweliereCheckout/pages/checkout/payment.html.twig';

        $clientSecret = $stripe->createPaymentIntent($session->getOrder());


        return $this->render($template, [
            "clientSecret"=> $clientSecret,
            "items" =>  $this->getSessionOrderItems()
        ]);
    }

    /**
     * @Route("/success", name="success", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function success(Request $request, TrackingDataProvider $tracking, OrderHandler $orderHandler): Response
    {
        $template = '@JuweliereCheckout/pages/checkout/success.html.twig';

        /** @var string $sessionId */
        $stripeId = $request->query->get("session_id");

        if (is_null($stripeId === false)) {
            $order = $this->sessionHandler->getOrder();
        } else {
            $order = $orderHandler->getByStripeCheckoutSessionId($stripeId);
        }

        $response = $this->render($template, [
            "tracking" => $tracking->getPurchaseData($order),
            "items" => $order->getItems(),
            "user" => $order->getUser()
        ]);

        return $response;
    }

    /**
     * @Route("/webhook/stripe", name="webhook_stripe", options={"expose"=true})
     *
     * @param Request $request
     * @param Environment $twig
     * @param \Juweliere\CheckoutBundle\Checkout\Mailer $mailer
     * @param LoggerInterface $logger
     * @param ParameterBagInterface $parameters,
     * @return Response
     */
    public function webhook(Request $request, Environment $twig, \Juweliere\CheckoutBundle\Checkout\Mailer $mailer, LoggerInterface $logger): Response
    {
        $disclaimerFile = $this->get('kernel')->getProjectDir() . DIRECTORY_SEPARATOR . "public/pdf/Widerrufsbelehrung_Vogl.pdf";


         // FOR STRIPE CLI TESTING
               /*\Stripe\Stripe::setApiKey($_SERVER["STRIPE_API_KEY"]);
               $session = Session::retrieve("cs_test_Vpca7SrfacBsgRbvedhRVxBWScK3g0AXIbOkEGFzGVSGFBpuVPvvSZWq");
               $paymentIntent = \Stripe\PaymentIntent::retrieve($session->payment_intent);

               #var_dump($session);
               #var_dump($customer);
               echo json_encode($paymentIntent->charges->data[0]);
               $mailer = new Mailer($mailer, $twig,  $request->getBaseUrl() . "cdn/");
               var_dump($mailer->sendCheckoutConfirmation($request, 1, $session, $paymentIntent->charges->data[0], $disclaimerFile));

               die();*/


        $payload = @file_get_contents('php://input');
        $event = null;

        try {
            $event = \Stripe\Event::constructFrom(
                json_decode($payload, true)
            );
        } catch(\UnexpectedValueException $e) {
            // Invalid payload
            http_response_code(400);
            exit();
        }

        try {
            // Handle the event
            switch ($event->type) {
                case 'checkout.session.completed':
                    /** @var Session $session */
                    $session = $event->data->object;

                    \Stripe\Stripe::setApiKey($_SERVER["STRIPE_API_KEY"]);

                    /** @var PaymentIntent $paymentIntent */
                    $paymentIntent = \Stripe\PaymentIntent::retrieve($session->payment_intent);

                    /** @var Charge $charge */
                    $charge = $paymentIntent->charges->data[0];

                    $entityManager = $this->getDoctrine()->getManager();

                    /** @var OrderStripe $order */
                    $order = $entityManager
                        ->getRepository(OrderStripe::class)
                        ->findOneBy(array("stripe_id" => $session->id));

                    $order->setData($session->toJSON());

                    $entityManager->persist($order);
                    $entityManager->flush();

                    try {
                        $mailer->sendCheckoutConfirmation(
                            $request,
                            $order->getId(),
                            $session,
                            $charge,
                            $disclaimerFile
                        );

                        $mailer->sendCheckoutOrder(
                            $request,
                            $order,
                            $session,
                            $charge,
                            $disclaimerFile
                        );
                    }catch (\Throwable $e) {
                        var_dump($e->getMessage());
                        http_response_code(500);
                        exit();
                    }

                    $logger->info($session->id);
                    break;
                default:
                    // Unexpected event type
                    http_response_code(400);
                    exit();
            }

            http_response_code(200);

            return $this->json(["result" => true]);
        } catch (\Exception $e) {
            var_dump($e->getMessage());
            http_response_code(400);
            exit();
        }

    }

    /**
     * @return Item[]|array
     */
    private function getSessionOrderItems() {

        /** @var Order $order */
        $order = $this->sessionHandler->getOrder();


        return $order->getItems();
    }
}
