<?php

namespace Juweliere\CheckoutBundle\Checkout;

use App\Entity\OrderStripe;
use Stripe\Charge;
use Stripe\Checkout\Session;
use Stripe\Customer;
use Swift_Message;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\HttpFoundation\Request;
use Twig\Environment;

class Mailer
{
    /**
     * @var Environment
     */
    private $twig;

    /**
     * @var \Swift_Mailer
     */
    private $mailer;

    /**
     * @var ParameterBagInterface
     */
    private $parameters;

    /**
     * @var string
     */
    private $baseUrlImage;

    /**
     * @var string
     */
    private $subjectPrefix = "";

    /**
     * @var string
     */
    private $publicPath;
    private $mailFrom;
    private $mailFromName;
    private $mailTo;
    private $baseUrlProduct;

    /**
     * @param \Swift_Mailer $mailer
     * @param Environment $twig
     * @param $mailFrom
     * @param $mailFromName
     * @param $mailTo
     * @param $baseUrlImage
     */
    public function __construct(
        \Swift_Mailer $mailer,
        Environment $twig,
        $mailFrom,
        $mailFromName,
        $mailTo,
        $baseUrlImage,
        $baseUrlProduct
    )
    {
        $this->mailer = $mailer;
        $this->twig = $twig;

        $this->mailFrom = $mailFrom;
        $this->mailFromName = $mailFromName;
        $this->mailTo = $mailTo;
        $this->baseUrlImage = $baseUrlImage;
        $this->baseUrlProduct = $baseUrlProduct;

        $logger = new \Swift_Plugins_Loggers_ArrayLogger();
        $this->mailer->registerPlugin(new \Swift_Plugins_LoggerPlugin($logger));

        $transport = $this->mailer->getTransport();
        if($transport instanceof \Swift_Transport_EsmtpTransport){
            $transport->setStreamOptions([
                'ssl' => [
                    'allow_self_signed' => true,
                    'verify_peer' => false,
                    'verify_peer_name' => false
                ]
            ]);
        }

        if (isset($_SERVER["MAIL_SUBJECT_PREFIX"])) {
            $this->subjectPrefix = $_SERVER["MAIL_SUBJECT_PREFIX"] . " ";
        } elseif (isset($_SERVER["APP_ENV"]) && strtolower($_SERVER["APP_ENV"]) != "prod") {
            $this->subjectPrefix = "(!!! " . $_SERVER["APP_ENV"] . " !!!) ";
        }
    }

    /**
     * @param Request $request
     * @param Session $session
     * @param Customer $customer
     * @param $disclaimerFilePath
     * @return int
     * @throws \Twig\Error\LoaderError
     * @throws \Twig\Error\RuntimeError
     * @throws \Twig\Error\SyntaxError
     */
    public function sendCheckoutOrder($orderId, $order, $disclaimerFilePath)
    {
        $order["orderId"] = $orderId;
        $order["productUrl"] = $this->baseUrlProduct;

        $mailTo = $this->mailTo;
        $mailToName = $this->mailFromName;

        if (trim($this->subjectPrefix) !== "") {
            $mailTo = $order["user"]["email"];
            $mailToName = $order["user"]["lastName"];
        }

        /** @var Swift_Message $message */
        $message = (new Swift_Message($this->subjectPrefix . 'Bestellung Online Shop'))
            ->setFrom($this->mailFrom, $this->mailFromName)
            ->setTo($mailTo, $mailToName)
            ->setBody($this->twig->render(
                '@JuweliereCheckout/mail/checkout/order.html.twig',
                $order
            ),
                'text/html');

        return $this->mailer->send($message);
    }
    /**
     * @param Request $request
     * @param Session $session
     * @param Customer $customer
     * @param $disclaimerFilePath
     * @return int
     * @throws \Twig\Error\LoaderError
     * @throws \Twig\Error\RuntimeError
     * @throws \Twig\Error\SyntaxError
     */
    public function sendUserInfo($orderId, $order, $subject, $template, $disclaimerFilePath = null)
    {
        /*
        var_dump($this->mailFrom);
        var_dump($this->mailFromName);
        var_dump($order["user"]);
        var_dump($order["user"]["email"]);
        */

        /** @var Swift_Message $message */
        $message = (new Swift_Message($this->subjectPrefix . $subject))
            ->setFrom($this->mailFrom, $this->mailFromName)
            ->setTo($order["user"]["email"], $order["user"]["lastName"])
            ->setBody($this->twig->render(
                '@JuweliereCheckout/mail/checkout/' . $template . '.html.twig',
                $order
            ),
                'text/html');

        if ($disclaimerFilePath !== null) {
            $message->attach(\Swift_Attachment::fromPath($disclaimerFilePath));
        }

        return $this->mailer->send($message);
    }

    /**
     * @param $email
     * @param $name
     * @return int
     * @throws \Twig\Error\LoaderError
     * @throws \Twig\Error\RuntimeError
     * @throws \Twig\Error\SyntaxError
     */
    public function sendTest($email, $name)
    {
        /** @var Swift_Message $message */
        $message = (new Swift_Message('Test Mail'))
            ->setFrom($this->mailFrom,$this->mailFromName)
            ->setTo($email, $name)
            ->setBody($this->twig->render(
                'mail/test.html.twig'
            ),
                'text/html');
        return $this->mailer->send($message);
    }
}
