<?php

declare(strict_types=1);

namespace Juweliere\CheckoutBundle\Controller;

use Juweliere\ApiBundle\Service\JuwApi;
use Juweliere\CheckoutBundle\Checkout\Order\Item;
use Juweliere\CheckoutBundle\Checkout\Order\Order;
use Juweliere\CheckoutBundle\Checkout\OrderHandler;
use Juweliere\CheckoutBundle\Checkout\ProductDataProvider;
use Juweliere\CheckoutBundle\Checkout\SessionHandler;
use Juweliere\CheckoutBundle\Checkout\StripeHandler;
use Juweliere\CheckoutBundle\Checkout\TrackingDataProvider;
use Juweliere\CheckoutBundle\Entity\OrderStripe;
use Psr\Log\LoggerInterface;
use Stripe\Charge;
use Stripe\Checkout\Session;
use Stripe\PaymentIntent;
use Symfony\Component\DependencyInjection\Container;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Generator\UrlGenerator;
use Twig\Environment;
use Sulu\Bundle\WebsiteBundle\Controller\WebsiteController;
use Symfony\Component\Routing\Annotation\Route;

/**
 * @Route("/checkout", requirements={"_locale": "en|es|fr"}, name="juweliere_checkout_checkout_")
 */
class CheckoutController extends WebsiteController
{
    const PRODUCT_TYPE_WATCH = "uhren";
    const PRODUCT_TYPE_JEWELLERY = "schmuck";
    const SESSION_LAST_SEEN = "lastseen";

    private SessionHandler $sessionHandler;
    private ProductDataProvider $productDataProvider;
    private bool $isProd = true;
    private ParameterBagInterface $parameterBag;

    /**
     * @param SessionHandler $sessionHandler
     * @param ProductDataProvider $productDataProvider
     */
    public function __construct(
        SessionHandler $sessionHandler,
        ProductDataProvider $productDataProvider,
        ParameterBagInterface $appParams
    ) {
        $this->sessionHandler = $sessionHandler;
        $this->productDataProvider = $productDataProvider;
        $this->isProd = $appParams->get('kernel.environment') === 'prod';
        $this->parameterBag = $appParams;
    }

    /**
     * @Route("/test123", name="test123", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function test(Request $request, JuwApi $juwApi): Response
    {
        var_dump(json_encode($juwApi->getMe()));
        die();
    }

    /**
     * @Route("/", name="basket", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function basket(Request $request): Response
    {
        $template = '@JuweliereCheckout/pages/checkout/basket.html.twig';

        $response = $this->render(
            $template,
            array (
                "items" => $this->getSessionOrderItems()
            )
        );

        return $response;
    }

    /**
     * @Route("/sku/{sku}", name="sku", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function sku(Request $request, string $sku, SessionHandler $sessionHandler): Response
    {
        $sessionHandler->addItemToOrder($sku);

        return $this->redirect("/checkout");
    }

    /**
     * @Route("/address", name="address", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function address(Request $request): Response
    {
        $template = '@JuweliereCheckout/pages/checkout/address.html.twig';

        $user = $this->getSessionUserData();
        $address = null;
        if ($user) {
            foreach($user->getAddresses() as $userAddress) {
                $address = $userAddress;
                break;
            }
        }

        $response = $this->render(
            $template,
            array (
                "items" => $this->getSessionOrderItems(),
                'user' => $user,
                'address' => $address
            )
        );

        return $response;
    }

    private function getPrefillData()
    {

    }

    /**
     * @Route("/stripetest/{sku}", name="stripetest", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function stripetest(Request $request, $sku, ProductDataProvider $productDataProvider, StripeHandler $stripe): Response
    {
        if (!$this->isProd) {
            return $this->redirect("/checkout");
        }

        return $this->stripe($request, $sku, $productDataProvider, $stripe, 111, "!!! TEST !!! ");
    }

    /**
     * @Route("/stripe/{sku}", name="stripe", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function stripe(Request $request, string $sku, ProductDataProvider $productDataProvider, StripeHandler $stripe, $price = null, $prefix = ""): Response
    {
        /** @var string $template */
        $template = '@JuweliereCheckout/pages/checkout/stripe.html.twig';

        /** @var array $product */
        $product = $productDataProvider->getProductBySku($sku);

        if (!is_null($price) && !$this->isProd) {
            $product["price"] =  $price;
        }

        $product["name"] = $prefix . $product["name"];
        $product["shortDescription"] = $prefix . $product["shortDescription"];

        $sessionId = $stripe->createCheckoutSession($product);

        $response = $this->render($template, ["session"=> $sessionId]);

        return $response;
    }

    /**
     * @Route("/payment", name="payment", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function payment(Request $request, SessionHandler $session, StripeHandler $stripe): Response
    {
        $this->sessionHandler->updateUserData($request, $this->getUser());

        $template = '@JuweliereCheckout/pages/checkout/payment.html.twig';

        $clientSecret = $stripe->createPaymentIntent($session->getOrder());

        return $this->render($template, [
            "clientSecret"=> $clientSecret,
            "items" =>  $this->getSessionOrderItems()
        ]);
    }

    /**
     * @Route("/success", name="success", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function success(Request $request, TrackingDataProvider $tracking, OrderHandler $orderHandler): Response
    {
        $template = '@JuweliereCheckout/pages/checkout/success.html.twig';

        /** @var string $sessionId */
        $stripeId = $request->query->get("session_id");

        if (is_null($stripeId)) {
            $order = $this->sessionHandler->getOrder();
        } else {
            $order = $orderHandler->getByStripeCheckoutSessionId($stripeId);
        }

        if (!$order) {
             return $this->redirectToRoute('juweliere_checkout_checkout_basket');
        }

        $this->sessionHandler->unsetOrder();

        $response = $this->render($template, [
            "tracking" => $tracking->getPurchaseData($order),
            "items" => $order->getItems(),
            "user" => $order->getUser(),
            "product" => $order->getItems()[0]
        ]);

        return $response;
    }

    /**
     * @Route("/webhook/mail", name="webhook_mail", options={"expose"=true})
     *
     * @param Request $request
     * @param Environment $twig
     * @param \Juweliere\CheckoutBundle\Checkout\Mailer $mailer
     * @param LoggerInterface $logger
     * @return Response
     */
    public function webhook(Request $request, Environment $twig, \Juweliere\CheckoutBundle\Checkout\Mailer $mailer, LoggerInterface $logger): Response
    {
        $disclaimerFile = $this->parameterBag->get('kernel.project_dir') . DIRECTORY_SEPARATOR . "public/pdf/Widerrufsbelehrung_Vogl.pdf";

        $order = json_decode($request->getContent(), true);

        foreach ($order["items"] as $key => $item) {
            $product = $this->productDataProvider->getProductBySku($item["sku"]);
            $item["image"] = $product["images"][0];
            $order["items"][$key] = $item;
        }

        #var_dump($order);

        try {
            $mailer->sendCheckoutConfirmation(
                $order["uuid"],
                $order,
                $disclaimerFile
            );

            $mailer->sendCheckoutOrder(
                $order["uuid"],
                $order,
                $disclaimerFile
            );
        }catch (\Throwable $e) {
            var_dump($e->getMessage());
            http_response_code(500);
            exit();
        }

        return new JsonResponse(array("result" => true));

    }

    /**
     * @return Item[]|array
     */
    private function getSessionOrderItems() {

        /** @var Order $order */
        $order = $this->sessionHandler->getOrder();

        if (is_null($order)) {
            return null;
        }

        return $order->getItems();
    }

    /**
     * @return Item[]|array
     */
    private function getSessionUserData() {
        return $this->getUser();
    }
}
