<?php

declare(strict_types=1);

namespace Juweliere\CheckoutBundle\Controller;

use Juweliere\ApiBundle\Service\JuwApi;
use Juweliere\CheckoutBundle\Checkout\Order\Item;
use Juweliere\CheckoutBundle\Checkout\Order\Order;
use Juweliere\CheckoutBundle\Checkout\OrderHandler;
use Juweliere\CheckoutBundle\Checkout\PaypalHandler;
use Juweliere\CheckoutBundle\Checkout\ProductDataProvider;
use Juweliere\CheckoutBundle\Checkout\SessionHandler;
use Juweliere\CheckoutBundle\Checkout\StripeHandler;
use Juweliere\CheckoutBundle\Checkout\TrackingDataProvider;
use Juweliere\CheckoutBundle\Entity\OrderStripe;
use Juweliere\CheckoutBundle\Handler\WebhookHandler;
use Juweliere\JuwApiClient\Constants;
use Juweliere\ProductBundle\Decorator\DecoratorApi;
use Psr\Log\LoggerInterface;
use Stripe\Charge;
use Stripe\Checkout\Session;
use Stripe\PaymentIntent;
use Symfony\Component\DependencyInjection\Container;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\Routing\Generator\UrlGenerator;
use Twig\Environment;
use Sulu\Bundle\WebsiteBundle\Controller\WebsiteController;
use Symfony\Component\Routing\Annotation\Route;

/**
 * @Route("/checkout", requirements={"_locale": "en|es|fr"}, name="juweliere_checkout_checkout_")
 */
class CheckoutController extends WebsiteController
{
    const PRODUCT_TYPE_WATCH = "uhren";
    const PRODUCT_TYPE_JEWELLERY = "schmuck";
    const SESSION_LAST_SEEN = "lastseen";

    private SessionHandler $sessionHandler;
    private ProductDataProvider $productDataProvider;
    private bool $isProd = true;
    private ParameterBagInterface $parameterBag;
    private DecoratorApi $decoratorApi;
    private Session $session;

    /**
     * @param SessionHandler $sessionHandler
     * @param ProductDataProvider $productDataProvider
     */
    public function __construct(
        SessionHandler $sessionHandler,
        ProductDataProvider $productDataProvider,
        ParameterBagInterface $appParams,
        DecoratorApi $decoratorApi
    ) {
        $this->sessionHandler = $sessionHandler;
        $this->productDataProvider = $productDataProvider;
        $this->isProd = $appParams->get('kernel.environment') === 'prod';
        $this->parameterBag = $appParams;
        $this->decoratorApi = $decoratorApi;
    }

    /**
     * @Route("/test123", name="test123", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function test(Request $request, JuwApi $juwApi): Response
    {
        var_dump(json_encode($juwApi->getMe()));
        die();
    }

    /**
     * @Route("/", name="basket", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function basket(Request $request): Response
    {
        $template = '@JuweliereCheckout/pages/checkout/basket.html.twig';

        if (!$this->getUser()) {
            $this->sessionHandler->setLoginRedirect("/checkout/address");
        }

        $response = $this->render(
            $template,
            array (
                "items" => $this->getSessionOrderItems()
            )
        );

        return $response;
    }



    /**
     * @Route("/basketcount", name="basket_count", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function basketCount(Request $request): Response
    {
        $count = 0;
        if (!is_null($this->sessionHandler->getOrder())) {
            $count = count($this->sessionHandler->getOrder()->getItems());
        }

        return new JsonResponse(array("count" => $count));
    }

    /**
     * @Route("/sku/{sku}", name="item_sku", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function sku(Request $request, string $sku, SessionHandler $sessionHandler): Response
    {
        $sessionHandler->addItemToOrder($sku);

        return $this->redirect("/checkout");
    }

    /**
     * @Route("/add/{sku}", name="item_add", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function addToCart(Request $request, string $sku, SessionHandler $sessionHandler, DecoratorApi $decoratorApi): Response
    {
        $sessionHandler->addItemToOrder($sku);
        $decoratorApi->createBasket($sku);

        return new JsonResponse(array("result" => true));
    }

    /**
     * @Route("/remove/{sku}", name="item_remove", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function removeFromCart(Request $request, string $sku, SessionHandler $sessionHandler, DecoratorApi $decoratorApi): Response
    {
        $sessionHandler->removeItemFromOrder($sku);
        $decoratorApi->removeBasket($sku);

        return $this->redirect($this->generateUrl("juweliere_checkout_checkout_basket"));
    }

    /**
     * @Route("/address", name="address", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function address(Request $request): Response
    {
        $template = '@JuweliereCheckout/pages/checkout/address.html.twig';

        $user = $this->getSessionUserData();
        $address = null;

        if ($user) {
            foreach($user->getAddresses() as $userAddress) {
                $address = $userAddress;
                break;
            }
        }

        $response = $this->render(
            $template,
            array (
                "items" => $this->getSessionOrderItems(),
                'user' => $user,
                'address' => $address,
                'addresses' => $user->getAddresses(),
                'countries' => $this->getCountries()
            )
        );

        return $response;
    }

    private function getPrefillData()
    {

    }

    /**
     * @Route("/stripetest/{sku}", name="stripetest", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function stripetest(Request $request, $sku, ProductDataProvider $productDataProvider, StripeHandler $stripe): Response
    {
        if (!$this->isProd) {
            return $this->redirect("/checkout");
        }

        return $this->stripe($request, $sku, $productDataProvider, $stripe, 111, "!!! TEST !!! ");
    }

    /**
     * @Route("/stripe/{sku}", name="stripe", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function stripe(Request $request, string $sku, ProductDataProvider $productDataProvider, StripeHandler $stripe, $price = null, $prefix = ""): Response
    {
        /** @var string $template */
        $template = '@JuweliereCheckout/pages/checkout/stripe.html.twig';

        /** @var array $product */
        $product = $productDataProvider->getProductBySku($sku);

        if (!is_null($price) && !$this->isProd) {
            $product["price"] =  $price;
        }

        $product["name"] = $prefix . $product["name"];
        $product["shortDescription"] = $prefix . $product["shortDescription"];

        $sessionId = $stripe->createCheckoutSession($product);

        $response = $this->render($template, ["session"=> $sessionId]);

        return $response;
    }

    /**
     * @Route("/payment/type", name="payment_type", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function paymentType(Request $request, PaypalHandler $paypal): Response
    {
        $order = $paypal->createOrder($this->sessionHandler->getOrder());

        return new JsonResponse($order);
    }

    /**
     * @Route("/payment", name="payment", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function payment(Request $request, StripeHandler $stripe): Response
    {
        try {
            $template = '@JuweliereCheckout/pages/checkout/payment.html.twig';

            $payerId = $request->get("payerId");

            if ($request->getMethod() === "POST" && !is_null($payerId)) {
                $paypal->confirmOrder($session->getOrder(), $payerId);
                return $this->redirect($this->generateUrl("juweliere_checkout_checkout_success"));
            }

            $this->sessionHandler->updateUserData($request, $this->getUser());

            if (!$this->getUser()) {
                $this->sessionHandler->setLoginRedirect("/checkout/address");
            }

            $order = $stripe->createPaymentIntent($this->sessionHandler->getOrder());

            $this->sessionHandler->setOrderId($order->getId());

            $billingAddress = $this->sessionHandler->getOrder()->getBillingAddress();

            if (empty($billingAddress) || empty($billingAddress->getGender())) {
                $billingAddress = $this->sessionHandler->getOrder()->getShippingAddress();
            }

            return $this->render($template, [
                "clientSecret"=> $order->getLastPayment()->getClientSecret(),
                "items" =>  $this->getSessionOrderItems(),
                "user" => $this->sessionHandler->getOrder()->getUser(),
                "shippingAddress" => $this->sessionHandler->getOrder()->getShippingAddress(),
                "billingAddress" => $billingAddress,
                "order" => $order,
                "countries" => $this->getCountries(),
                "pickupInStore" => $this->sessionHandler->getOrder()->getPickupInStore()
            ]);
        } catch (\Throwable $e) {
            var_dump($e->getMessage());
            die();
            return $this->redirect($this->generateUrl("juweliere_checkout_checkout_basket"));
        }
    }

    private function getCountries() {
        return array(
            array(
                "code" => "de",
                "label" => "Deutschland"
            ),
            array(
                "code" => "at",
                "label" => "Österreich"
            ),
            array(
                "code" => "ch",
                "label" => "Schweiz"
            )
        );
    }

    /**
     * @Route("/success", name="success", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function success(Request $request, TrackingDataProvider $tracking, OrderHandler $orderHandler): Response
    {
        $template = '@JuweliereCheckout/pages/checkout/success.html.twig';

        /** @var string $sessionId */
        $stripeId = $request->query->get("session_id");

        if (is_null($stripeId)) {
            $order = $this->sessionHandler->getOrder();
        } else {
            $order = $orderHandler->getByStripeCheckoutSessionId($stripeId);
        }

        if (!$order) {
             return $this->redirectToRoute('juweliere_checkout_checkout_basket');
        }

        $this->sessionHandler->unsetOrder();

        $response = $this->render($template, [
            "tracking" => $tracking->getPurchaseData($order),
            "items" => $order->getItems(),
            "user" => $order->getUser(),
            "product" => $order->getItems()[0],
            "order" => $order
        ]);

        $this->decoratorApi->createCheckout($order);

        return $response;
    }

    /**
     * @Route("/webhook/mail", name="webhook_mail", options={"expose"=true})
     *
     * @param Request $request
     * @param Environment $twig
     * @param \Juweliere\CheckoutBundle\Checkout\Mailer $mailer
     * @param LoggerInterface $logger
     * @return Response
     */
    public function webhook(Request $request, Environment $twig, WebhookHandler $webhookHandler): Response
    {
        $disclaimerFile = $this->parameterBag->get('kernel.project_dir') . DIRECTORY_SEPARATOR . "public/pdf/Widerrufsbelehrung_Vogl.pdf";

        $order = json_decode($request->getContent(), true);

        foreach ($order["items"] as $key => $item) {
            $product = $this->productDataProvider->getProductBySku($item["sku"]);
            $item["image"] = $product["images"][0];
            $order["items"][$key] = $item;
        }

        #var_dump($order);

        try {
            $webhookHandler->mail($order, $disclaimerFile);
        }catch (\Throwable $e) {
            var_dump($e->getMessage());
            http_response_code(500);
            exit();
        }

        return new JsonResponse(array("result" => true));

    }

    /**
     * @return Item[]|array
     */
    private function getSessionOrderItems() {

        /** @var Order $order */
        $order = $this->sessionHandler->getOrder();

        if (is_null($order)) {
            return null;
        }

        return $order->getItems();
    }

    /**
     * @return Item[]|array
     */
    private function getSessionUserData() {
        return $this->getUser();
    }
}
