<?php

namespace Juweliere\CheckoutBundle\Checkout;

use App\Entity\OrderStripe;
use Stripe\Charge;
use Stripe\Checkout\Session;
use Stripe\Customer;
use Symfony\Bridge\Twig\Mime\TemplatedEmail;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Component\Mime\Address;
use Twig\Environment;

class Mailer
{
    /**
     * @var Environment
     */
    private $twig;

    /**
     * @var MailerInterface
     */
    private $mailer;

    /**
     * @var ParameterBagInterface
     */
    private $parameters;

    /**
     * @var string
     */
    private $baseUrlImage;

    /**
     * @var string
     */
    private $subjectPrefix = "";

    /**
     * @var string
     */
    private $publicPath;
    private $mailFrom;
    private $mailFromName;
    private $mailTo;
    private $baseUrlProduct;
    private $baseUrlDecorator;

    /**
     * @param MailerInterface $mailer
     * @param Environment $twig
     * @param $mailFrom
     * @param $mailFromName
     * @param $mailTo
     * @param $baseUrlImage
     */
    public function __construct(
        MailerInterface $mailer,
        Environment $twig,
                        $mailFrom,
                        $mailFromName,
                        $mailTo,
                        $baseUrlImage,
                        $baseUrlProduct,
                        $baseUrlDecorator
    )
    {
        $this->mailer = $mailer;
        $this->twig = $twig;

        $this->mailFrom = $mailFrom;
        $this->mailFromName = $mailFromName;
        $this->mailTo = $mailTo;
        $this->baseUrlImage = $baseUrlImage;
        $this->baseUrlProduct = $baseUrlProduct;
        $this->baseUrlDecorator = $baseUrlDecorator;

        if (isset($_SERVER["MAIL_SUBJECT_PREFIX"])) {
            $this->subjectPrefix = $_SERVER["MAIL_SUBJECT_PREFIX"] . " ";
        } elseif (isset($_SERVER["APP_ENV"]) && strtolower($_SERVER["APP_ENV"]) != "prod") {
            $this->subjectPrefix = "(!!! " . $_SERVER["APP_ENV"] . " !!!) ";
        }
    }

    /**
     * @param Request $request
     * @param Session $session
     * @param Customer $customer
     * @param $filePath
     * @return int
     * @throws \Twig\Error\LoaderError
     * @throws \Twig\Error\RuntimeError
     * @throws \Twig\Error\SyntaxError
     */
    public function sendCheckoutOrder($orderId, $order, $filePath)
    {
        $order["orderId"] = $orderId;
        $order["productUrl"] = $this->baseUrlProduct;
        $order["decoratorUrl"] = $this->baseUrlDecorator;

        $mailTo = $this->mailTo;
        $mailToName = $this->mailFromName;

        if (trim($this->subjectPrefix) !== "") {
            $mailTo = $order["user"]["email"];
            $mailToName = $order["user"]["lastName"];
        }

        /** @var TemplatedEmail $message */
        $message = (new TemplatedEmail())
            ->subject($this->subjectPrefix . 'Bestellung Online Shop')
            ->from(new Address($this->mailFrom, $this->mailFromName))
            ->to($mailTo)
            ->htmlTemplate('@JuweliereCheckout/mail/checkout/order.html.twig')
            ->context($order);

        try {
            $this->mailer->send($message);
            return true;
        } catch (\Throwable $e) {
//            var_dump($e->getMessage());
            return false;
        }
    }
    /**
     * @param Request $request
     * @param Session $session
     * @param Customer $customer
     * @param $filePath
     * @return int
     * @throws \Twig\Error\LoaderError
     * @throws \Twig\Error\RuntimeError
     * @throws \Twig\Error\SyntaxError
     */
    public function sendUserInfo($orderId, $order, $subject, $template, $sendCopy = true, $filePath = null)
    {
        /*
        var_dump($this->mailFrom);
        var_dump($this->mailFromName);
        var_dump($order["user"]);
        var_dump($order["user"]["email"]);
        */

        /** @var TemplatedEmail $message */
        $message = (new TemplatedEmail())
            ->subject($this->subjectPrefix . $subject)
            ->from(new Address($this->mailFrom, $this->mailFromName))
            ->to($order["user"]["email"])
            ->htmlTemplate('@JuweliereCheckout/mail/checkout/' . $template . '.html.twig')
            ->context($order);

        if ($filePath !== null && file_exists($filePath)) {
            $message->attachFromPath($filePath);
        }

        try {
            $this->mailer->send($message);

            if ($sendCopy) {
                /** @var TemplatedEmail $message */

                $mailTo = $this->mailTo;

                if (trim($this->subjectPrefix) !== "") {
                    $mailTo = $order["user"]["email"];
                }

                $order["decoratorUrl"] = $this->baseUrlDecorator;
                $order["showCopyHint"] = true;
                $message = (new TemplatedEmail())
                    ->subject($this->subjectPrefix . 'KOPIE - ' . $subject)
                    ->from(new Address($this->mailFrom, $this->mailFromName))
                    ->to($mailTo)
                    ->htmlTemplate('@JuweliereCheckout/mail/checkout/' . $template . '.html.twig')
                    ->context($order);
                $this->mailer->send($message);
            }

            return true;
        } catch (\Throwable $e) {
//            var_dump($e->getMessage());
            return false;
        }

        return true;
    }

    /**
     * @param $email
     * @param $name
     * @return int
     * @throws \Twig\Error\LoaderError
     * @throws \Twig\Error\RuntimeError
     * @throws \Twig\Error\SyntaxError
     */
    public function sendTest($email, $name)
    {

        /** @var TemplatedEmail $message */
        $message = (new TemplatedEmail())
            ->subject($this->subjectPrefix . 'Bestellung Online Shop')
            ->from(new Address($this->mailFrom, $this->mailFromName))
            ->to($email)
            ->htmlTemplate('mail/test.html.twig');

        try {
            $this->mailer->send($message);
            return true;
        } catch (\Throwable $e) {
//            var_dump($e->getMessage());
            return false;
        }
    }
}
