<?php


namespace Juweliere\CheckoutBundle\Checkout;

use Juweliere\ProductBundle\ElasticSearch\ElasticProductsParser;
use Juweliere\ProductBundle\Service\ProductDataHelper;

class ProductDataProvider
{
    private ElasticProductsParser $productsParser;
    private ProductDataHelper $productDataHelper;
    private string $imageBaseUrl;

    /**
     * ProductDataProvider constructor.
     * @param ElasticProductsParser $productsParser
     * @param ProductDataHelper $productDataHelper
     * @param string $imageBaseUrl
     */
    public function __construct(
        ElasticProductsParser $productsParser,
        ProductDataHelper     $productDataHelper,
        string                $imageBaseUrl
    )
    {
        $this->productsParser = $productsParser;
        $this->productDataHelper = $productDataHelper;
        $this->imageBaseUrl = $imageBaseUrl;
    }

    /**
     * @param $sku
     * @return array|null
     * @throws \Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface
     * @throws \Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface
     * @throws \Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface
     * @throws \Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface
     * @throws \Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface
     */
    public function getProductBySku($sku)
    {
        $product = $this->productsParser->getSingleProduct($sku);

        if (is_null($product)) {
            return null;
        }

        $product = $this->productDataHelper->consolidateGalleryImages($product);

        $product["images"] = $this->getFullPathImages($product);
        $product["price"] = $this->getPrice($product);

        return $product;
    }

    /**
     * @param array $product
     * @return float|int|null
     * @throws \Exception
     */
    private function getPrice(array $product): float|int|null
    {
        $price = $this->productDataHelper->getPrice($product);

        if (!$price) {
            return null;
        }

        return $price["actualPrice"]["value"] * 100;
    }

    /**
     * @param array $product
     * @return array
     */
    private function getFullPathImages(array $product, $dimension = "250x250")
    {
        /** @var array $images */
        $images = array();

        if (!empty($product["gallery"])) {
            foreach ($product["gallery"] as $image) {
                $images[] = $this->imageBaseUrl . $dimension . "/" . $image["path"];
            }
        } else {
            $images[] = $this->imageBaseUrl . $dimension . "/" . $product["image"];
        }

        return $images;
    }
}
