<?php

declare(strict_types=1);

namespace Juweliere\CheckoutBundle\Controller;

use App\Service\NewsletterService;
use Juweliere\ApiBundle\Service\JuwApi;
use Juweliere\CheckoutBundle\Checkout\Order\Address;
use Juweliere\CheckoutBundle\Checkout\Order\Item;
use Juweliere\CheckoutBundle\Checkout\Order\Order;
use Juweliere\CheckoutBundle\Checkout\OrderHandler;
use Juweliere\CheckoutBundle\Checkout\PaypalHandler;
use Juweliere\CheckoutBundle\Checkout\ProductDataProvider;
use Juweliere\CheckoutBundle\Checkout\SessionHandler;
use Juweliere\CheckoutBundle\Checkout\StripeHandler;
use Juweliere\CheckoutBundle\Checkout\TrackingDataProvider;
use Juweliere\CheckoutBundle\Checkout\UserHandler;
use Juweliere\CheckoutBundle\Entity\OrderStripe;
use Juweliere\CheckoutBundle\Handler\WebhookHandler;
use Juweliere\JuwApiClient\Constants;
use Juweliere\JuwApiClient\Entity\User;
use Juweliere\ProductBundle\Decorator\DecoratorClient;
use Psr\Log\LoggerInterface;
use Stripe\Charge;
use Stripe\Checkout\Session;
use Stripe\PaymentIntent;
use Symfony\Component\DependencyInjection\Container;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\Routing\Generator\UrlGenerator;
use Twig\Environment;
use Sulu\Bundle\WebsiteBundle\Controller\WebsiteController;
use Symfony\Component\Routing\Annotation\Route;

/**
 * @Route("/checkout", requirements={"_locale": "en|es|fr"}, name="juweliere_checkout_checkout_")
 */
class CheckoutController extends WebsiteController
{
    const PRODUCT_TYPE_WATCH = "uhren";
    const PRODUCT_TYPE_JEWELLERY = "schmuck";
    const SESSION_LAST_SEEN = "lastseen";

    private SessionHandler $sessionHandler;
    private ProductDataProvider $productDataProvider;
    private bool $isProd = true;
    private ParameterBagInterface $parameterBag;
    private DecoratorClient $decoratorClient;
    private Session $session;
    private PaypalHandler $paypalHandler;
    private JuwApi $juwApi;
    private NewsletterService $newsletterService;

    /**
     * @param SessionHandler $sessionHandler
     * @param ProductDataProvider $productDataProvider
     */
    public function __construct(
        SessionHandler $sessionHandler,
        ProductDataProvider $productDataProvider,
        ParameterBagInterface $appParams,
        DecoratorClient $decoratorClient,
        PaypalHandler $paypalHandler,
        NewsletterService $newsletterService,
        JuwApi $juwApi
    ) {
        $this->sessionHandler = $sessionHandler;
        $this->productDataProvider = $productDataProvider;
        $this->isProd = $appParams->get('kernel.environment') === 'prod';
        $this->parameterBag = $appParams;
        $this->decoratorClient = $decoratorClient;
        $this->paypalHandler = $paypalHandler;
        $this->juwApi = $juwApi;
        $this->newsletterService = $newsletterService;
    }

    /**
     * @Route("/test123", name="test123", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function test(Request $request, JuwApi $juwApi): Response
    {
        var_dump(json_encode($juwApi->getMe()));
        die();
    }

    /**
     * @Route("/", name="basket", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function basket(Request $request): Response
    {
        $template = '@JuweliereCheckout/pages/checkout/basket.html.twig';

        if (!$this->getUser()) {
            $this->sessionHandler->setLoginRedirect("/checkout/address");
        }

        $this->sessionHandler->unsetOrderTypeGuest();

        $response = $this->render(
            $template,
            array (
                "items" => $this->getSessionOrderItems()
            )
        );

        return $response;
    }

    /**
     * @Route("/basket", name="basket_items", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function basketItems(Request $request): Response
    {
        $template = '@JuweliereCheckout/snippets/slide-menu/basketitems.html.twig';

        if (!$this->getUser()) {
            $this->sessionHandler->setLoginRedirect("/checkout/address");
        }

        $this->sessionHandler->unsetOrderTypeGuest();

        $response = $this->renderView(
            $template,
            array (
                "items" => $this->getSessionOrderItems()
            )
        );

        return new JsonResponse(["content" => $response]);
    }



    /**
     * @Route("/basketcount", name="basket_count", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function basketCount(Request $request): Response
    {
        $count = 0;
        if (!is_null($this->sessionHandler->getOrder())) {
            $count = count($this->sessionHandler->getOrder()->getItems());
        }

        return new JsonResponse(array("count" => $count));
    }

    /**
     * @Route("/sku/{sku}", name="item_sku", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function sku(Request $request, string $sku, SessionHandler $sessionHandler): Response
    {
        $sessionHandler->addItemToOrder($sku);

        return $this->redirect("/checkout");
    }

    /**
     * @Route("/status/{sku}", name="status_item", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function status(Request $request, string $sku): Response
    {
        $product = $this->productDataProvider->getProductBySku($sku);
        $isSaleable = false;
        $isInBasket = false;

        if ($product["isEnabled"] && $product["isVisible"] && $product["isSaleable"] && $product["qty"] > 0) {
            $isSaleable = true;
        }

        try {
            $basketItems =  $this->getSessionOrderItems();

            /** @var Item $basketItem */
            foreach ($basketItems as $basketItem) {
                if ($basketItem->getSku() === $sku) {
                    $isInBasket = true;
                    break;
                }
            }

        } catch (\Throwable $e) {

        }

        $response =  new JsonResponse(
            array(
                "isSaleable" => $isSaleable,
                "isInBasket" => $isInBasket
            )
        );

        $response->setMaxAge(0);
        $response->setSharedMaxAge(0);
        $response->headers->set('Cache-Control', 'no-cache, must-revalidate');

        return $response;
    }

    /**
     * @Route("/add/{sku}", name="item_add", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function addToCart(Request $request, string $sku, SessionHandler $sessionHandler): Response
    {
        $sessionHandler->addItemToOrder($sku);
        $this->decoratorClient->createBasket($sku);

        return new JsonResponse(array("result" => true));
    }

    /**
     * @Route("/remove/{sku}", name="item_remove", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function removeFromCart(Request $request, string $sku, SessionHandler $sessionHandler): Response
    {
        $sessionHandler->removeItemFromOrder($sku);
        $this->decoratorClient->removeBasket($sku);

        return $this->redirect($this->generateUrl("juweliere_checkout_checkout_basket"));
    }

    /**
     * @Route("/address", name="address", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function address(Request $request): Response
    {
        if (is_null($this->sessionHandler->getOrder())) {
            return $this->redirect($this->generateUrl("juweliere_checkout_checkout_basket"));
        }

        /** @var User $user */
        $user = $this->getSessionUserData();

        $guestModeAllowed = boolval($this->parameterBag->get('guest_mode'));
        $phoneRequired = boolval($this->parameterBag->get('checkout_phone_required'));

        if (!is_null($request->get("guest")) && $request->get("guest") === "1" && $guestModeAllowed) {
            $this->sessionHandler->setOrderTypeGuest();
        } elseif (!$this->sessionHandler->getOrderTypeGuest() && is_null($user)) {
            return $this->redirect($this->generateUrl("juweliere_user_user_login", ["_target_path" => "/checkout/address", "guest" => true]));
        }

        $template = '@JuweliereCheckout/pages/checkout/address.html.twig';
        $address = null;
        $adresses = null;

        if ($user) {
            $billingAddress = $this->sessionHandler->getOrder()->getBillingAddress();
            if (!is_null($billingAddress)) {
                $shippingAddress = $this->sessionHandler->getOrder()->getShippingAddress();
            } else {
                if (!$user->getAddresses()) {
                    $billingAddress = array(
                        "firstName" => $user->getFirstName(),
                        "lastName" => $user->getLastName()
                    );
                } else {
                    foreach ($user->getAddresses() as $userAddress) {
                        $billingAddress = $userAddress;
                        break;
                    }
                }
                $shippingAddress = $billingAddress;
            }
            $adresses = $user->getAddresses();
        }

        $data = array (
            "isGuest" => $this->sessionHandler->getOrderTypeGuest(),
            "items" => $this->getSessionOrderItems(),
            'user' => $user,
            'billing_address' => $billingAddress,
            'shipping_address' => $shippingAddress,
            'addresses' => $adresses,
            'countries' => $this->getCountries(),
            'phoneRequired' => $phoneRequired
        );

        $newUser = $this->get('session')->get('_new_user');

        if ($newUser) {
            $this->get('session')->set('_new_user', false);
            $data["newUser"] = true;
        }

        $response = $this->render(
            $template,
            $data
        );

        return $response;
    }

    private function getPrefillData()
    {

    }

    /**
     * @Route("/stripetest/{sku}", name="stripetest", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function stripetest(Request $request, $sku, ProductDataProvider $productDataProvider, StripeHandler $stripe): Response
    {
        if (!$this->isProd) {
            return $this->redirect("/checkout");
        }

        return $this->stripe($request, $sku, $productDataProvider, $stripe, 111, "!!! TEST !!! ");
    }

    /**
     * @Route("/stripe/{sku}", name="stripe", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function stripe(Request $request, string $sku, ProductDataProvider $productDataProvider, StripeHandler $stripe, $price = null, $prefix = ""): Response
    {
        /** @var string $template */
        $template = '@JuweliereCheckout/pages/checkout/stripe.html.twig';

        /** @var array $product */
        $product = $productDataProvider->getProductBySku($sku);

        if (!is_null($price) && !$this->isProd) {
            $product["price"] =  $price;
        }

        $product["name"] = $prefix . $product["name"];
        $product["shortDescription"] = $prefix . $product["shortDescription"];

        $sessionId = $stripe->createCheckoutSession($product);

        $response = $this->render($template, ["session"=> $sessionId]);

        return $response;
    }


    /**
     * @Route("/payment/type", name="payment_type", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function paymentType(Request $request, StripeHandler $stripe): Response
    {
        if (strtoupper($request->get("type")) === JuwApi::PAYMENT_TYPE_MOLLIE) {
            $token = $request->get("token");
            if (is_null($token)) {
                return new JsonResponse(["error" => "missing token"]);
            }
            $order = $stripe->createPaymentIntent($this->sessionHandler->getOrder(), JuwApi::PAYMENT_TYPE_MOLLIE, $token);
            $this->sessionHandler->setOrderId($order->getId());
        } else {
            $order = $this->paypalHandler->createOrder($this->sessionHandler->getOrder());
        }

        return new JsonResponse($order);
    }

    /**
     * @Route("/order/update", name="order_update", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function udpate(Request $request, StripeHandler $stripeHandler): Response
    {
        $message = trim($request->get("message"));
        $result = true;

        $newsletter = boolval($request->get("newsletter"));

        if (!empty($message) || $newsletter) {
            if ($this->sessionHandler->updateOrderData($request)) {
                try {
                    $order = $this->sessionHandler->getOrder();
                    $order->setPayment(null);

                    $order = $this->juwApi->orders->update($order);
                } catch (\Throwable $e) {
//                    return new JsonResponse(["result" => false, "error" => $e->getMessage()], 400);
                }
            };
        }

        return new JsonResponse(["result" => $result]);
    }

    /**
     * @Route("/payment", name="payment", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function payment(Request $request, StripeHandler $stripe): Response
    {
        if (is_null($this->sessionHandler->getOrder())) {
            return $this->redirect($this->generateUrl("juweliere_checkout_checkout_basket"));
        }

        try {
            $template = '@JuweliereCheckout/pages/checkout/payment.html.twig';

            $this->sessionHandler->updateUserData($request, $this->getUser());

            if (!$this->getUser()) {
                $this->sessionHandler->setLoginRedirect("/checkout/address");
            }

            $type = $this->parameterBag->get("payment_type");

            $clientSecret = "";

            if ($type == JuwApi::PAYMENT_TYPE_MOLLIE) {
                $sessionOrder = $this->sessionHandler->getOrder();
                $sessionOrder->setPayment(null);
                $sessionOrder->setId(null);
                $order = $this->juwApi->orders->create($sessionOrder);
            } else {
                $order = $stripe->createPaymentIntent($this->sessionHandler->getOrder(), $type);
                $clientSecret = $order->getLastPayment()->getClientSecret();
            }

            $this->sessionHandler->setOrderId($order->getId());

            $billingAddress = $this->sessionHandler->getOrder()->getBillingAddress();

            if (empty($billingAddress) || empty($billingAddress->getGender())) {
                $billingAddress = $this->sessionHandler->getOrder()->getShippingAddress();
            }

            return $this->render($template, [
                "isGuest" => $this->sessionHandler->getOrderTypeGuest(),
                "clientSecret"=> $clientSecret,
                "items" =>  $this->getSessionOrderItems(),
                "user" => $this->sessionHandler->getOrder()->getUser(),
                "shippingAddress" => $this->sessionHandler->getOrder()->getShippingAddress(),
                "billingAddress" => $billingAddress,
                "order" => $order,
                "countries" => $this->getCountries(),
                "pickupInStore" => $this->sessionHandler->getOrder()->getPickupInStore(),
                "pickupStoreId" => $this->sessionHandler->getOrder()->getPickupStoreId()
            ]);
        } catch (\Throwable $e) {
//            dd($e->getMessage(), $e->getTraceAsString());
            return $this->redirect($this->generateUrl("juweliere_checkout_checkout_basket"));
        }
    }

    private function getCountries() {

        $availableCountries =  array(
            array(
                "code" => "de",
                "label" => "Deutschland"
            ),
            array(
                "code" => "at",
                "label" => "Österreich"
            ),
            array(
                "code" => "ch",
                "label" => "Schweiz"
            )
        );

        $countries = explode(",", $this->parameterBag->get('checkout_countries'));

        $displayCountries = array();

        foreach ($availableCountries as $availableCountry) {
            if (in_array($availableCountry["code"], $countries)) {
                $displayCountries[] = $availableCountry;
            }
        }

        return $displayCountries;
    }

    /**
     * @Route("/success", name="success", options={"expose"=true})
     *
     * @param Request $request
     * @return Response
     *
     */
    public function success(Request $request, TrackingDataProvider $tracking, OrderHandler $orderHandler, UserHandler $userHandler): Response
    {
        $template = '@JuweliereCheckout/pages/checkout/success.html.twig';

        /** @var string $sessionId */
        $stripeId = $request->query->get("session_id");
        $orderId = $request->query->get("id");
        $newsletter = boolval($request->get("newsletter"));
        $checkCounter = intval($request->get("count"));

        $isError = false;
        $isPaymentError = false;
        $checkStatus = false;

        $orderStatus = \Juweliere\CheckoutBundle\Constants::ORDER_STATUS_INIT;

        if (is_null($stripeId)) {
            $order = $this->sessionHandler->getOrder();
        } else {
            $order = $orderHandler->getByStripeCheckoutSessionId($stripeId);
        }

        try {
            if (!is_null($orderId)) {
                $checkStatus = true;
                $checkCounter++;
                $orderStatus = $this->juwApi->orders->get($orderId)->getStatus();
            }
        } catch (\Throwable $e) {
            $isError = true;
        }


        if (!$order || $isError) {
            return $this->redirectToRoute('juweliere_checkout_checkout_basket');
        }

        $paymentFailedStatus = [
            \Juweliere\CheckoutBundle\Constants::ORDER_STATUS_PAYMENT_FAILED,
            \Juweliere\CheckoutBundle\Constants::ORDER_STATUS_PAYMENT_CANCELED,
            \Juweliere\CheckoutBundle\Constants::ORDER_STATUS_PAYMENT_EXPIRED
        ];

        if ($checkStatus && $orderStatus === \Juweliere\CheckoutBundle\Constants::ORDER_STATUS_INIT) {
            if($checkCounter < 3) {
                sleep(2);
                return $this->redirectToRoute('juweliere_checkout_checkout_success', ["id" => $orderId, "count" => $checkCounter]);
            }
            $isPaymentError = true;
        }

        if($isPaymentError || in_array($orderStatus, $paymentFailedStatus)) {
            return $this->redirectToRoute('juweliere_checkout_checkout_payment', ["error" => 1]);
        };

        if ($newsletter) {
            try {
                $this->newsletterService->createFromOrder($order);
            } catch (\Throwable $e) {

            }
        }

        if ($request->getMethod() === "POST") {
            $payerId = $request->get("payerId");

            if (is_null($payerId)) {
                return $this->redirect($this->generateUrl("juweliere_checkout_checkout_payment"));
            }

            try {
                $this->paypalHandler->confirmOrder($order, $payerId);
            } catch (\Throwable $e) {
                return $this->redirect($this->generateUrl("juweliere_checkout_checkout_payment"));
            }
        }

        if (!$this->sessionHandler->getOrderTypeGuest()) {
            try {
                if ($this->sessionHandler->getStoreBillingAddress()) {
                    $userHandler->addAddress($this->sessionHandler->getOrder()->getBillingAddress(), $this->getUser());
                }
                if ($this->sessionHandler->getStoreShippingAddress()) {
                    $userHandler->addAddress($this->sessionHandler->getOrder()->getShippingAddress(), $this->getUser());
                }
            } catch (\Throwable $e) {
            }
        }

        $this->sessionHandler->unsetOrder();

        $response = $this->render($template, [
            "tracking" => $tracking->getPurchaseData($order),
            "items" => $order->getItems(),
            "user" => $order->getUser(),
            "product" => $order->getItems()[0],
            "order" => $order
        ]);

        $this->decoratorClient->createCheckout($order->getId(), $order->getItems());

        return $response;
    }

    /**
     * @Route("/webhook/mail", name="webhook_mail", options={"expose"=true})
     *
     * @param Request $request
     * @param Environment $twig
     * @param \Juweliere\CheckoutBundle\Checkout\Mailer $mailer
     * @return Response
     */
    public function webhook(Request $request, Environment $twig, WebhookHandler $webhookHandler): Response
    {
        $disclaimerFile = $this->parameterBag->get('kernel.project_dir') . DIRECTORY_SEPARATOR . "public/pdf/Widerrufsbelehrung_Vogl.pdf";

        $order = json_decode($request->getContent(), true);

        foreach ($order["items"] as $key => $item) {
            $product = $this->productDataProvider->getProductBySku($item["sku"]);
            $item["image"] = $product["images"][0];
            $item["brand"] = $product["brand"];
            $order["items"][$key] = $item;
        }

        try {
            $webhookHandler->mail($order, $disclaimerFile);
        }catch (\Throwable $e) {
            var_dump($e->getMessage());
            http_response_code(500);
            exit();
        }

        return new JsonResponse(array("result" => true));

    }

    /**
     * @Route("/webhook/mail/reminder", name="webhook_mail_reminder", options={"expose"=true})
     *
     * @param Request $request
     * @param Environment $twig
     * @param \Juweliere\CheckoutBundle\Checkout\Mailer $mailer
     * @return Response
     */
    public function webhookreminder(Request $request, Environment $twig, WebhookHandler $webhookHandler): Response
    {
        $orders = json_decode($request->getContent(), true);

        try {
            $webhookHandler->remindermail($orders);
        }catch (\Throwable $e) {
            var_dump($e->getMessage());
            http_response_code(500);
            exit();
        }

        return new JsonResponse(array("result" => true));

    }

    /**
     * @return Item[]|array
     */
    private function getSessionOrderItems() {

        /** @var Order $order */
        $order = $this->sessionHandler->getOrder();

        if (is_null($order)) {
            return null;
        }

        return $order->getItems();
    }

    /**
     * @return Item[]|array
     */
    private function getSessionUserData() {
        return $this->getUser();
    }
}
