<?php

namespace Juweliere\CheckoutBundle\Checkout;

use Twig\Error\LoaderError;
use Twig\Error\RuntimeError;
use Twig\Error\SyntaxError;
use Stripe\Checkout\Session;
use Stripe\Customer;
use Symfony\Bridge\Twig\Mime\TemplatedEmail;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Component\Mime\Address;

class Mailer
{
    private string $subjectPrefix = "";

    /**
     * @param $mailFrom
     * @param $mailFromName
     * @param $mailTo
     * @param $baseUrlImage
     */
    public function __construct(
        private readonly MailerInterface $mailer,
                        private $mailFrom,
                        private $mailFromName,
                        private $mailTo,
                        private $baseUrlProduct,
                        private $baseUrlDecorator
    )
    {
        if (isset($_SERVER["MAIL_SUBJECT_PREFIX"])) {
            $this->subjectPrefix = $_SERVER["MAIL_SUBJECT_PREFIX"] . " ";
        } elseif (isset($_SERVER["APP_ENV"]) && strtolower((string) $_SERVER["APP_ENV"]) != "prod") {
            $this->subjectPrefix = "(!!! " . $_SERVER["APP_ENV"] . " !!!) ";
        }
    }

    /**
     * @param Request $request
     * @param Session $session
     * @param Customer $customer
     * @param $filePath
     * @throws LoaderError
     * @throws RuntimeError
     * @throws SyntaxError
     */
    public function sendCheckoutOrder($orderId, array $order, $filePath): bool
    {
        $order["orderId"] = $orderId;
        $order["productUrl"] = $this->baseUrlProduct;
        $order["decoratorUrl"] = $this->baseUrlDecorator;

        $mailTo = $this->mailTo;
        $mailToName = $this->mailFromName;

        if (trim($this->subjectPrefix) !== "") {
            $mailTo = $order["user"]["email"];
            $mailToName = $order["user"]["lastName"];
        }

        /** @var TemplatedEmail $message */
        $message = (new TemplatedEmail())
            ->subject($this->subjectPrefix . 'Bestellung Online Shop')
            ->from(new Address($this->mailFrom, $this->mailFromName))
            ->to($mailTo)
            ->htmlTemplate('@JuweliereCheckout/mail/checkout/order.html.twig')
            ->context($order);

        try {
            $this->mailer->send($message);
            return true;
        } catch (\Throwable) {
//            var_dump($e->getMessage());
            return false;
        }
    }

    /**
     * @param Request $request
     * @param Session $session
     * @param Customer $customer
     * @param $filePath
     * @throws LoaderError
     * @throws RuntimeError
     * @throws SyntaxError
     */
    public function sendUserInfo($orderId, array $order, string $subject, string $template, $sendCopy = true, $filePath = null): bool
    {
        /*
        var_dump($this->mailFrom);
        var_dump($this->mailFromName);
        var_dump($order["user"]);
        var_dump($order["user"]["email"]);
        */

        /** @var TemplatedEmail $message */
        $message = (new TemplatedEmail())
            ->subject($this->subjectPrefix . $subject)
            ->from(new Address($this->mailFrom, $this->mailFromName))
            ->to($order["user"]["email"])
            ->htmlTemplate('@JuweliereCheckout/mail/checkout/' . $template . '.html.twig')
            ->context($order);

        if ($filePath !== null && file_exists($filePath)) {
            $message->attachFromPath($filePath);
        }

        try {
            $this->mailer->send($message);

            if ($sendCopy) {
                /** @var TemplatedEmail $message */

                $mailTo = $this->mailTo;

                if (trim($this->subjectPrefix) !== "") {
                    $mailTo = $order["user"]["email"];
                }

                $order["decoratorUrl"] = $this->baseUrlDecorator;
                $order["showCopyHint"] = true;
                $message = (new TemplatedEmail())
                    ->subject($this->subjectPrefix . 'KOPIE - ' . $subject)
                    ->from(new Address($this->mailFrom, $this->mailFromName))
                    ->to($mailTo)
                    ->htmlTemplate('@JuweliereCheckout/mail/checkout/' . $template . '.html.twig')
                    ->context($order);
                $this->mailer->send($message);
            }

            return true;
        } catch (\Throwable) {
//            var_dump($e->getMessage());
            return false;
        }
    }

    /**
     * @param $email
     * @param $name
     * @throws LoaderError
     * @throws RuntimeError
     * @throws SyntaxError
     */
    public function sendReminder($orders): bool
    {
        /** @var TemplatedEmail $message */
        $message = (new TemplatedEmail())
            ->subject($this->subjectPrefix . 'Bestellungen erfordern Handlung')
            ->from(new Address($this->mailFrom, $this->mailFromName))
            ->to($this->mailTo)
            ->htmlTemplate('@JuweliereCheckout/mail/reminder/reminder.html.twig')
            ->context(["orders" => $orders, "decoratorUrl" => $this->baseUrlDecorator]);

        try {
            $this->mailer->send($message);
            return true;
        } catch (\Throwable $throwable) {
            var_dump($throwable->getMessage());
            return false;
        }
    }

    /**
     * @param $email
     * @param $name
     * @throws LoaderError
     * @throws RuntimeError
     * @throws SyntaxError
     */
    public function sendTest(Address|string $email, $name): bool
    {

        /** @var TemplatedEmail $message */
        $message = (new TemplatedEmail())
            ->subject($this->subjectPrefix . 'Bestellung Online Shop')
            ->from(new Address($this->mailFrom, $this->mailFromName))
            ->to($email)
            ->htmlTemplate('mail/test.html.twig');

        try {
            $this->mailer->send($message);
            return true;
        } catch (\Throwable) {
//            var_dump($e->getMessage());
            return false;
        }
    }
}
