<?php

declare(strict_types=1);

namespace Juweliere\CheckoutBundle\Checkout;

use Juweliere\JuwApiClient\Entity\Address;
use Juweliere\JuwApiClient\Entity\Order\Item;
use Juweliere\JuwApiClient\Entity\Order\Order;
use Juweliere\JuwApiClient\Entity\User;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;

class SessionHandler
{
    const string SESSION_LOGIN_REDIRECT = "loginRedirect";

    const string SESSION_CHECKOUT_ORDER = "checkout.order";

    const string SESSION_CHECKOUT_GUEST = "checkout.guest";

    const string SESSION_STORE_BILLING_ADDRESS = "checkout.store.billing";

    const string SESSION_STORE_SHIPPING_ADDRESS = "checkout.store.shipping";

    public function __construct(
        private readonly RequestStack        $requestStack,
        private readonly ProductDataProvider $productDataProvider)
    {
    }

    public function setLoginRedirect($uri): void
    {
        $this->requestStack->getSession()->set(self::SESSION_LOGIN_REDIRECT, $uri);
    }

    /**
     * @throws TransportExceptionInterface
     * @throws ServerExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws ClientExceptionInterface
     */
    public function addItemToOrder(string $sku, int $quantity = 1): true
    {

        /** @var array $product */
        $product = $this->productDataProvider->getProductBySku($sku);

        $order = new Order();

        if ($this->requestStack->getSession()->has(self::SESSION_CHECKOUT_ORDER)) {
            $order = $this->requestStack->getSession()->get(self::SESSION_CHECKOUT_ORDER);
        }

        $item = new Item();

        $item->setSku($sku);
        $item->setName($product["name"]);
        $item->setShortDescription($product["shortDescription"]);
        $item->setPrice($product["price"]);
        $item->setQuantity($quantity);
        $item->setImages($product["images"]);
        $item->setImage($product["images"][0]);
        $item->setReference($product["reference"]);
        $item->setBrand($product["brand"]);

        $order->addItem($item);

        $this->requestStack->getSession()->set(self::SESSION_CHECKOUT_ORDER, $order);
        $this->requestStack->getSession()->save();

        return true;
    }

    public function removeItemFromOrder($sku): true
    {

        if ($this->requestStack->getSession()->has(self::SESSION_CHECKOUT_ORDER)) {
            $order = $this->requestStack->getSession()->get(self::SESSION_CHECKOUT_ORDER);
            $order->removeItem($sku);

            $this->requestStack->getSession()->set(self::SESSION_CHECKOUT_ORDER, $order);
            $this->requestStack->getSession()->save();
        }

        return true;
    }

    public function setOrderId($id): void
    {
        if ($this->requestStack->getSession()->has(self::SESSION_CHECKOUT_ORDER)) {
            /** @var Order $order */
            $order = $this->requestStack->getSession()->get(self::SESSION_CHECKOUT_ORDER);
            $order->setId($id);
            $this->requestStack->getSession()->set(self::SESSION_CHECKOUT_ORDER, $order);
        }
    }

    public function updateUserData(Request $request, UserInterface $sessionUser = null): void
    {
        if ($this->requestStack->getSession()->has(self::SESSION_CHECKOUT_ORDER) && !$request->isMethod("GET")) {
            /** @var Order $order */
            $order = $this->requestStack->getSession()->get(self::SESSION_CHECKOUT_ORDER);

            $user = new User();

            if ($sessionUser !== null) {
                $user->setId($sessionUser->getId());
            }

            $user->setGender($request->get("billing_gender"));
            $user->setFirstName($request->get("billing_firstName"));
            $user->setLastName($request->get("billing_lastName"));
            $user->setEmail($request->get("email"));

            $order->setUser($user);

            $billingAddress = new Address();

            $billingAddress->setGender($request->get("billing_gender"));
            $billingAddress->setFirstName($request->get("billing_firstName"));
            $billingAddress->setLastName($request->get("billing_lastName"));
            $billingAddress->setStreet($request->get("billing_street"));
            $billingAddress->setStreet2($request->get("billing_street2"));
            $billingAddress->setCity($request->get("billing_city"));
            $billingAddress->setState($request->get("billing_state"));
            $billingAddress->setCountry($request->get("billing_country"));
            $billingAddress->setPostcode($request->get("billing_postcode"));
            $billingAddress->setPhone($request->get("billing_phone"));

            $order->setBillingAddress($billingAddress);

            $pickupInStore = boolval($request->get("pickupInStore"));

            if ($pickupInStore) {
                $pickupStoreId = $request->get("pickupStoreId");
                if (!empty($pickupStoreId)) {
                    $order->setPickupStoreId($pickupStoreId);
                }
            }

            $order->setPickupInStore($pickupInStore);

            if ($pickupInStore) {
                $pickupStoreId = $request->get("pickupStoreId");
                if (!empty($pickupStoreId)) {
                    $order->setPickupStoreId($pickupStoreId);
                }
            }

            $giftPack = boolval($request->get("giftPack"));
            $order->setGiftPack($giftPack);
            $order->setShippingAddress($billingAddress);

            if (boolval($request->get("shippingAddress")) === true) {
                $shippingAddress = new Address();

                $shippingAddress->setGender($request->get("shipping_gender"));
                $shippingAddress->setFirstName($request->get("shipping_firstName"));
                $shippingAddress->setLastName($request->get("shipping_lastName"));
                $shippingAddress->setStreet($request->get("shipping_street"));
                $shippingAddress->setStreet2($request->get("shipping_street2"));
                $shippingAddress->setCity($request->get("shipping_city"));
                $shippingAddress->setState($request->get("shipping_state"));
                $shippingAddress->setCountry($request->get("shipping_country"));
                $shippingAddress->setPostcode($request->get("shipping_postcode"));
                $shippingAddress->setPhone($request->get("shipping_phone"));

                $order->setShippingAddress($shippingAddress);
            } elseif ($pickupInStore) {
                $order->setShippingAddress(null);
            }

            $storeBillingAddress = boolval($request->get("billing_address_store"));
            $storeShippingAddress = boolval($request->get("shipping_address_store"));

            $this->requestStack->getSession()->set(self::SESSION_STORE_BILLING_ADDRESS, $storeBillingAddress);
            $this->requestStack->getSession()->set(self::SESSION_STORE_SHIPPING_ADDRESS, $storeShippingAddress);

            /*echo "<pre>";
            var_dump($order);
            echo "</pre>";
            die();*/


            $this->requestStack->getSession()->set(self::SESSION_CHECKOUT_ORDER, $order);
        }
    }

    public function updateOrderData($request): bool
    {
        if ($this->requestStack->getSession()->has(self::SESSION_CHECKOUT_ORDER)) {
            $order = $this->requestStack->getSession()->get(self::SESSION_CHECKOUT_ORDER);

            $message = trim((string) $request->get("message"));

            if (!empty($message)) {
                $order->setMessage($message);
            }

            $newsletter = trim((string) $request->get("newsletter"));

            if (!empty($newsletter)) {
                $order->setNewsletter($newsletter);
            }

            $this->requestStack->getSession()->set(self::SESSION_CHECKOUT_ORDER, $order);

            return true;
        }

        return false;
    }

    public function getOrderTypeGuest(): bool
    {
        if ($this->requestStack->getSession()->get(self::SESSION_CHECKOUT_GUEST)) {
            return true;
        }

        return false;
    }

    public function setOrderTypeGuest($type = true): void
    {
        $this->requestStack->getSession()->set(self::SESSION_CHECKOUT_GUEST, $type);
    }

    public function unsetOrderTypeGuest($type = true): void
    {
        $this->requestStack->getSession()->remove(self::SESSION_CHECKOUT_GUEST);
    }

    public function unsetOrder(): void
    {
        $this->requestStack->getSession()->remove(self::SESSION_CHECKOUT_ORDER);
    }

    public function getStoreBillingAddress(): mixed
    {
        return $this->requestStack->getSession()->get(self::SESSION_STORE_BILLING_ADDRESS);
    }

    public function getStoreShippingAddress(): mixed
    {
        return $this->requestStack->getSession()->get(self::SESSION_STORE_SHIPPING_ADDRESS);
    }

    public function getOrder(): Order
    {
        return $this->requestStack->getSession()->get(self::SESSION_CHECKOUT_ORDER);
    }
}
