<?php


namespace Juweliere\CheckoutBundle\Checkout;

use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Juweliere\ProductBundle\ElasticSearch\ElasticProductsParser;
use Juweliere\ProductBundle\Service\ProductDataHelper;

class ProductDataProvider
{
    /**
     * ProductDataProvider constructor.
     */
    public function __construct(private readonly ElasticProductsParser $productsParser, private readonly ProductDataHelper     $productDataHelper, private readonly string                $imageBaseUrl)
    {
    }

    /**
     * @param $sku
     * @return array|null
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function getProductBySku(string $sku)
    {
        $product = $this->productsParser->getSingleProduct($sku);

        if (is_null($product)) {
            return null;
        }

        $product = $this->productDataHelper->consolidateGalleryImages($product);

        $product["images"] = $this->getFullPathImages($product);
        $product["price"] = $this->getPrice($product);

        return $product;
    }

    /**
     * @throws \Exception
     */
    private function getPrice(array $product): float|int|null
    {
        $price = $this->productDataHelper->getPrice($product);

        if (!$price) {
            return null;
        }

        return $price["actualPrice"]["value"] * 100;
    }

    /**
     * @return array
     */
    private function getFullPathImages(array $product, string $dimension = "250x250")
    {
        /** @var array $images */
        $images = [];

        if (!empty($product["gallery"])) {
            foreach ($product["gallery"] as $image) {
                $images[] = $this->imageBaseUrl . $dimension . "/" . $image["path"];
            }
        } else {
            $images[] = $this->imageBaseUrl . $dimension . "/" . $product["image"];
        }

        return $images;
    }
}
