<?php

declare(strict_types=1);

namespace Juweliere\CheckoutBundle\Handler;

use Juweliere\CheckoutBundle\Checkout\Mailer;
use Juweliere\CheckoutBundle\Constants;
use Symfony\Contracts\Translation\TranslatorInterface;
use Twig\Error\LoaderError;
use Twig\Error\RuntimeError;
use Twig\Error\SyntaxError;

class WebhookHandler
{
    public function __construct(private readonly Mailer $mailer, private readonly TranslatorInterface $translator)
    {
    }

    /**
     * @throws RuntimeError
     * @throws LoaderError
     * @throws SyntaxError
     */
    public function mail(array $order, $disclaimerFile): void
    {
        switch ($order["status"]) {
            case Constants::ORDER_STATUS_REFUNDED:
            case Constants::ORDER_STATUS_REFUNDED_NO_TRANSACTION:
                $this->mailer->sendUserInfo(
                    $order["id"],
                    $order,
                    $this->translator->trans("juw.checkout.mail.subject.refund"),
                    "refund"
                );
                break;

            case Constants::ORDER_STATUS_RETURNED:
                $this->mailer->sendUserInfo(
                    $order["id"],
                    $order,
                    $this->translator->trans("juw.checkout.mail.subject.return"),
                    "return"
                );
                break;

            case Constants::ORDER_STATUS_PAYMENT_SUCCEEDED:
                $this->mailer->sendUserInfo(
                    $order["id"],
                    $order,
                    $this->translator->trans("juw.checkout.mail.subject.confirmation"),
                    "confirmation",
                    false,
                    $disclaimerFile
                );

                $this->mailer->sendCheckoutOrder(
                    $order["id"],
                    $order,
                    $disclaimerFile
                );

                break;

            case Constants::ORDER_STATUS_SHIPPED:
                $this->mailer->sendUserInfo(
                    $order["id"],
                    $order,
                    $this->translator->trans("juw.checkout.mail.subject.shipped"),
                    "shipped"
                );

                break;

            case Constants::ORDER_PICKED_UP:
                $this->mailer->sendUserInfo(
                    $order["id"],
                    $order,
                    $this->translator->trans("juw.checkout.mail.subject.picked_up"),
                    "picked_up"
                );

                break;

            case Constants::ORDER_IN_PROGRESS:
                $this->mailer->sendUserInfo(
                    $order["id"],
                    $order,
                    $this->translator->trans("juw.checkout.mail.subject.in_progress"),
                    "in_progress"
                );

                break;

            case Constants::ORDER_STATUS_CANCELED:
                $this->mailer->sendUserInfo(
                    $order["id"],
                    $order,
                    $this->translator->trans("juw.checkout.mail.subject.canceled"),
                    "canceled"
                );

                break;

            case Constants::ORDER_STATUS_RETURN_ANNOUNCED:
                $this->mailer->sendUserInfo(
                    $order["id"],
                    $order,
                    $this->translator->trans("juw.checkout.mail.subject.return_announced"),
                    "return_announced"
                );

                break;
        }
    }

    /**
     * @throws SyntaxError
     * @throws RuntimeError
     * @throws LoaderError
     */
    public function remindermail($orders): void
    {
        $this->mailer->sendReminder($orders);
    }
}
