<?php

declare(strict_types=1);

namespace Juweliere\CheckoutBundle\Controller;

use Juweliere\JuwApiClient\Entity\Address;
use Juweliere\CheckoutBundle\Constants;
use App\Service\NewsletterService;
use JetBrains\PhpStorm\NoReturn;
use Juweliere\ApiBundle\Service\JuwApi;
use Juweliere\CheckoutBundle\Checkout\MollieHandler;
use Juweliere\CheckoutBundle\Checkout\Order\Item;
use Juweliere\CheckoutBundle\Checkout\Order\Order;
use Juweliere\CheckoutBundle\Checkout\OrderHandler;
use Juweliere\CheckoutBundle\Checkout\PayoneHandler;
use Juweliere\CheckoutBundle\Checkout\PaypalHandler;
use Juweliere\CheckoutBundle\Checkout\ProductDataProvider;
use Juweliere\CheckoutBundle\Checkout\SessionHandler;
use Juweliere\CheckoutBundle\Checkout\StripeHandler;
use Juweliere\CheckoutBundle\Checkout\TrackingDataProvider;
use Juweliere\CheckoutBundle\Checkout\UserHandler;
use Juweliere\CheckoutBundle\Handler\WebhookHandler;
use Juweliere\JuwApiClient\Entity\User;
use Juweliere\ProductBundle\Decorator\DecoratorClient;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Twig\Environment;
use Sulu\Bundle\WebsiteBundle\Controller\WebsiteController;
use Symfony\Component\Routing\Attribute\Route;
use Psr\Log\LoggerInterface;

#[Route("/checkout", name: "juweliere_checkout_checkout_", requirements: ["_locale" => "en|es|fr"])]
class CheckoutController extends WebsiteController
{
    const string PRODUCT_TYPE_WATCH = "uhren";

    const string PRODUCT_TYPE_JEWELLERY = "schmuck";

    const string SESSION_LAST_SEEN = "lastseen";

    private bool $isProd = true;

    public function __construct(
        private readonly SessionHandler        $sessionHandler,
        private readonly ProductDataProvider   $productDataProvider,
        private readonly ParameterBagInterface $parameterBag,
        private readonly DecoratorClient       $decoratorClient,
        private readonly PaypalHandler         $paypalHandler,
        private readonly PayoneHandler         $payoneHandler,
        private readonly NewsletterService     $newsletterService,
        private readonly JuwApi                $juwApi,
        private readonly LoggerInterface       $logger
    ) {
        $this->isProd = $this->parameterBag->get('kernel.environment') === 'prod';
    }

    #[NoReturn]
    #[Route("/test123", name: "test123", options: ["expose" => true])]
    public function test(JuwApi $juwApi): Response
    {
        var_dump(json_encode($juwApi->getMe()));
        die();
    }

    #[Route("/", name: "basket", options: ["expose" => true])]
    public function basket() : Response
    {
        $template = '@JuweliereCheckout/pages/checkout/basket.html.twig';
        if ($this->getUser() === null) {
            $this->sessionHandler->setLoginRedirect("/checkout/address");
        }
        $this->sessionHandler->unsetOrderTypeGuest();
        return $this->render(
            $template,
            ["items" => $this->getSessionOrderItems()]
        );
    }

    #[Route("/basket", name: "basket_items", options: ["expose" => true])]
    public function basketItems(): Response
    {
        $template = '@JuweliereCheckout/snippets/slide-menu/basketitems.html.twig';

        if ($this->getUser() === null) {
            $this->sessionHandler->setLoginRedirect("/checkout/address");
        }

        $this->sessionHandler->unsetOrderTypeGuest();

        $response = $this->renderView(
            $template,
            ["items" => $this->getSessionOrderItems()]
        );

        return new JsonResponse(["content" => $response]);
    }

    #[Route("/basketcount", name: "basket_count", options: ["expose" => true])]
    public function basketCount(): Response
    {
        $count = 0;
        if (!is_null($this->sessionHandler->getOrder())) {
            $count = count($this->sessionHandler->getOrder()->getItems());
        }

        return new JsonResponse(["count" => $count]);
    }

    #[Route("/sku/{sku}", name: "item_sku", options: ["expose" => true])]
    public function sku(string $sku, SessionHandler $sessionHandler): Response
    {
        $sessionHandler->addItemToOrder($sku);

        return $this->redirect("/checkout");
    }

    /**
     * @throws TransportExceptionInterface
     * @throws ServerExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws ClientExceptionInterface
     */
    #[Route("/status/{sku}", name: "status_item", options: ["expose" => true])]
    public function status(string $sku): Response
    {
        $product = $this->productDataProvider->getProductBySku($sku);
        $isSaleable = false;
        $isInBasket = false;

        if ($product["isEnabled"] && $product["isVisible"] && $product["isSaleable"] && $product["qty"] > 0) {
            $isSaleable = true;
        }

        try {
            $basketItems =  $this->getSessionOrderItems();

            /** @var Item $basketItem */
            foreach ($basketItems as $basketItem) {
                if ($basketItem->getSku() === $sku) {
                    $isInBasket = true;
                    break;
                }
            }

        } catch (\Throwable) {

        }

        $response =  new JsonResponse(
            ["isSaleable" => $isSaleable, "isInBasket" => $isInBasket]
        );

        $response->setMaxAge(0);
        $response->setSharedMaxAge(0);

        $response->headers->set('Cache-Control', 'no-cache, must-revalidate');

        return $response;
    }

    #[Route("/add/{sku}", name: "item_add", options: ["expose" => true])]
    public function addToCart(string $sku, SessionHandler $sessionHandler): Response
    {
        $sessionHandler->addItemToOrder($sku);
        $this->decoratorClient->createBasket($sku);

        return new JsonResponse(["result" => true]);
    }

    #[Route("/remove/{sku}", name: "item_remove", options: ["expose" => true])]
    public function removeFromCart(string $sku, SessionHandler $sessionHandler): Response
    {
        $sessionHandler->removeItemFromOrder($sku);
        $this->decoratorClient->removeBasket($sku);

        return $this->redirectToRoute("juweliere_checkout_checkout_basket");
    }

    #[Route("/address", name: "address", options: ["expose" => true])]
    public function address(Request $request): Response
    {
        if (is_null($this->sessionHandler->getOrder())) {
            return $this->redirectToRoute("juweliere_checkout_checkout_basket");
        }

        /** @var User $user */
        $user = $this->getSessionUserData();

        $guestModeAllowed = boolval($this->parameterBag->get('guest_mode'));
        $phoneRequired = boolval($this->parameterBag->get('checkout_phone_required'));

        if (!is_null($request->get("guest")) && $request->get("guest") === "1" && $guestModeAllowed) {
            $this->sessionHandler->setOrderTypeGuest();
        } elseif (!$this->sessionHandler->getOrderTypeGuest() && is_null($user)) {
            return $this->redirectToRoute("juweliere_user_user_login", ["_target_path" => "/checkout/address", "guest" => true]);
        }

        $template = '@JuweliereCheckout/pages/checkout/address.html.twig';
        $billingAddress = null;
        $shippingAddress = null;
        $addresses = null;

        $billingAddress = null;
        $shippingAddress = null;

        if ($user) {
            $billingAddress = $this->sessionHandler->getOrder()->getBillingAddress();
            if (!is_null($billingAddress)) {
                $shippingAddress = $this->sessionHandler->getOrder()->getShippingAddress();
            } else {
                if (!$user->getAddresses()) {
                    $billingAddress = ["firstName" => $user->getFirstName(), "lastName" => $user->getLastName()];
                } else {
                    foreach ($user->getAddresses() as $userAddress) {
                        $billingAddress = $userAddress;
                        break;
                    }
                }

                $shippingAddress = $billingAddress;
            }

            $addresses = $user->getAddresses();
        }

        $data = ["isGuest" => $this->sessionHandler->getOrderTypeGuest(), "items" => $this->getSessionOrderItems(), 'user' => $user, 'billing_address' => $billingAddress, 'shipping_address' => $shippingAddress, 'addresses' => $addresses, 'countries' => $this->getCountries(), 'phoneRequired' => $phoneRequired];

        $session = $request->getSession();
        $newUser = $session->get('_new_user');

        if ($newUser) {
            $session->set('_new_user', false);
            $data["newUser"] = true;
        }

        return $this->render(
            $template,
            $data
        );
    }

    /**
     * @throws TransportExceptionInterface
     * @throws ServerExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws ClientExceptionInterface
     */
    #[Route("/stripetest/{sku}", name: "stripetest", options: ["expose" => true])]
    public function stripetest(string $sku, ProductDataProvider $productDataProvider, StripeHandler $stripe): Response
    {
        if (!$this->isProd) {
            return $this->redirect("/checkout");
        }

        return $this->stripe($sku, $productDataProvider, $stripe, 111, "!!! TEST !!! ");
    }

    /**
     * @throws TransportExceptionInterface
     * @throws ServerExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws ClientExceptionInterface
     */
    #[Route("/stripe/{sku}", name: "stripe", options: ["expose" => true])]
    public function stripe(string $sku, ProductDataProvider $productDataProvider, StripeHandler $stripe, $price = null, string $prefix = ""): Response
    {
        $template = '@JuweliereCheckout/pages/checkout/stripe.html.twig';

        /** @var array $product */
        $product = $productDataProvider->getProductBySku($sku);

        if (!is_null($price) && !$this->isProd) {
            $product["price"] =  $price;
        }

        $product["name"] = $prefix . $product["name"];
        $product["shortDescription"] = $prefix . $product["shortDescription"];

        $sessionId = $stripe->createCheckoutSession($product);

        return $this->render($template, ["session"=> $sessionId]);
    }

    #[Route("/payment/type", name: "payment_type", options: ["expose" => true])]
    public function paymentType(Request $request, MollieHandler $mollieHandler): Response
    {
        if (strtoupper((string) $request->get("type", "")) === JuwApi::PAYMENT_TYPE_MOLLIE) {
            $token = $request->get("token");
            if (is_null($token)) {
                return new JsonResponse(["error" => "missing token"]);
            }

            $order = $mollieHandler->createPayment($this->sessionHandler->getOrder(), $token);
            $this->sessionHandler->setOrderId($order->getId());
        } else {
            $order = $this->paypalHandler->createOrder($this->sessionHandler->getOrder());
        }

        return new JsonResponse($order);
    }

    #[Route("/order/update", name: "order_update", options: ["expose" => true])]
    public function update(Request $request): Response
    {
        $message = trim((string) $request->get("message"));
        $result = true;

        $newsletter = boolval($request->get("newsletter"));

        if (!empty($message) || $newsletter) {
            if ($this->sessionHandler->updateOrderData($request)) {
                try {
                    $order = $this->sessionHandler->getOrder();
                    $order->setPayment(null);

                    $order = $this->juwApi->orders->update($order);
                } catch (\Throwable) {
//                    return new JsonResponse(["result" => false, "error" => $e->getMessage()], 400);
                }
            };
        }

        return new JsonResponse(["result" => $result]);
    }

    #[Route("/payment/redirect/{type}", name: "payment_redirect", options: ["expose" => true])]
    public function paymentRedirect(Request $request, PayoneHandler $payoneHandler, $type): Response
    {
        $this->sessionHandler->updateOrderData($request);

        try {
            $order = $this->sessionHandler->getOrder();

            if (strtoupper($type) === "PAYONE") {
                $order = $payoneHandler->createOrder($order);
            } else {
                $order->setPayment(strtoupper($type));
                $order = $this->juwApi->orders->create($order);
            }

            $this->sessionHandler->setOrderId($order->getId());

            if (!empty($order->getLastPayment()->getRedirect())) {
                return $this->redirect($order->getLastPayment()->getRedirect());
            }

            $this->logger->error("PAYMENT REDIRECT ERROR: missing redirect url in api response");
        } catch (\Throwable $e) {
            $this->logger->error("PAYMENT REDIRECT ERROR: " . $e->getMessage());
        }

        return $this->redirect($this->generateUrl("juweliere_checkout_checkout_overview"));
    }

    #[Route("/overview", name: "overview", options: ["expose" => true])]
    public function overview(Request $request): Response
    {
        if (is_null($this->sessionHandler->getOrder())) {
            return $this->redirect($this->generateUrl("juweliere_checkout_checkout_basket"));
        }

        try {
            $template = '@JuweliereCheckout/pages/checkout/overview.html.twig';

            $this->sessionHandler->updateUserData($request, $this->getUser());

            if (!$this->getUser()) {
                $this->sessionHandler->setLoginRedirect("/checkout/address");
            }

            $clientSecret = "";

            $sessionOrder = $this->sessionHandler->getOrder();
            $sessionOrder->setPayment(null);
            $sessionOrder->setId(null);
            $order = $sessionOrder;

            $billingAddress = $this->sessionHandler->getOrder()->getBillingAddress();

            if (empty($billingAddress) || empty($billingAddress->getGender())) {
                $billingAddress = $this->sessionHandler->getOrder()->getShippingAddress();
            }

            return $this->render($template, [
                "isGuest" => $this->sessionHandler->getOrderTypeGuest(),
                "clientSecret"=> $clientSecret,
                "items" =>  $this->getSessionOrderItems(),
                "user" => $this->sessionHandler->getOrder()->getUser(),
                "shippingAddress" => $this->sessionHandler->getOrder()->getShippingAddress(),
                "billingAddress" => $billingAddress,
                "order" => $order,
                "countries" => $this->getCountries(),
                "pickupInStore" => $this->sessionHandler->getOrder()->getPickupInStore(),
                "pickupStoreId" => $this->sessionHandler->getOrder()->getPickupStoreId()
            ]);
        } catch (\Throwable $e) {
            return $this->redirect($this->generateUrl("juweliere_checkout_checkout_address"));
        }
    }

    #[Route("/payment", name: "payment", options: ["expose" => true])]
    public function payment(Request $request, StripeHandler $stripe): Response
    {
        if (is_null($this->sessionHandler->getOrder())) {
            return $this->redirectToRoute("juweliere_checkout_checkout_basket");
        }

        try {
            $template = '@JuweliereCheckout/pages/checkout/payment.html.twig';

            $this->sessionHandler->updateUserData($request, $this->getUser());

            if ($this->getUser() === null) {
                $this->sessionHandler->setLoginRedirect("/checkout/address");
            }

            $type = $this->parameterBag->get("payment_type");

            $clientSecret = "";

            if ($type == JuwApi::PAYMENT_TYPE_MOLLIE) {
                $sessionOrder = $this->sessionHandler->getOrder();
                $sessionOrder->setPayment(null);
                $sessionOrder->setId(null);
                $order = $this->juwApi->orders->create($sessionOrder);
            } else {
                $order = $stripe->createPaymentIntent($this->sessionHandler->getOrder(), $type);
                $clientSecret = $order->getLastPayment()->getClientSecret();
            }

            $this->sessionHandler->setOrderId($order->getId());

            $billingAddress = $this->sessionHandler->getOrder()->getBillingAddress();

            if (!$billingAddress instanceof Address || empty($billingAddress->getGender())) {
                $billingAddress = $this->sessionHandler->getOrder()->getShippingAddress();
            }

            return $this->render($template, [
                "isGuest" => $this->sessionHandler->getOrderTypeGuest(),
                "clientSecret"=> $clientSecret,
                "items" =>  $this->getSessionOrderItems(),
                "user" => $this->sessionHandler->getOrder()->getUser(),
                "shippingAddress" => $this->sessionHandler->getOrder()->getShippingAddress(),
                "billingAddress" => $billingAddress,
                "order" => $order,
                "countries" => $this->getCountries(),
                "pickupInStore" => $this->sessionHandler->getOrder()->getPickupInStore(),
                "pickupStoreId" => $this->sessionHandler->getOrder()->getPickupStoreId()
            ]);
        } catch (\Throwable) {
            return $this->redirectToRoute("juweliere_checkout_checkout_basket");
        }
    }

    private function getCountries(): array
    {
        $availableCountries = [
            [
                "code" => "de",
                "label" => "Deutschland"
            ],
            [
                "code" => "at",
                "label" => "Österreich"
            ],
            [
                "code" => "ch",
                "label" => "Schweiz"
            ],
        ];

        $countries = explode(",", $this->parameterBag->get('checkout_countries'));

        $displayCountries = [];

        foreach ($availableCountries as $availableCountry) {
            if (in_array($availableCountry["code"], $countries)) {
                $displayCountries[] = $availableCountry;
            }
        }

        return $displayCountries;
    }

    #[Route("/success", name: "success", options: ["expose" => true])]
    public function success(
        Request $request,
        TrackingDataProvider $tracking,
        OrderHandler $orderHandler,
        UserHandler $userHandler
    ): Response
    {
        $template = '@JuweliereCheckout/pages/checkout/success.html.twig';

        $stripeId = $request->query->get("session_id");
        $payone = $request->query->get("payone");
        $orderId = $request->query->get("id");
        $newsletter = boolval($request->get("newsletter"));
        $checkCounter = intval($request->get("count"));

        $isError = false;
        $isPaymentError = false;
        $checkStatus = false;

        $orderStatus = Constants::ORDER_STATUS_INIT;

        if (is_null($stripeId)) {
            $order = $this->sessionHandler->getOrder();
        } else {
            $order = $orderHandler->getByStripeCheckoutSessionId($stripeId);
        }

        if(!is_null($payone)) {
            $orderId = $payone;
        }

        try {
            if (!is_null($orderId)) {
                $checkStatus = true;
                $checkCounter++;
                $orderStatus = $this->juwApi->orders->get($orderId)->getStatus();
            }
        } catch (\Throwable) {
            $isError = true;
        }



        if (!$order || $isError) {
            return $this->redirectToRoute('juweliere_checkout_checkout_basket');
        }

        $paymentFailedStatus = [
            Constants::ORDER_STATUS_PAYMENT_FAILED,
            Constants::ORDER_STATUS_PAYMENT_CANCELED,
            Constants::ORDER_STATUS_PAYMENT_EXPIRED
        ];

        if ($checkStatus && $orderStatus === Constants::ORDER_STATUS_INIT) {
            if($checkCounter < 3) {
                sleep(2);
                return $this->redirectToRoute('juweliere_checkout_checkout_success', ["id" => $orderId, "count" => $checkCounter]);
            }

            $isPaymentError = true;
        }

        if($isPaymentError || in_array($orderStatus, $paymentFailedStatus)) {
            return $this->redirectToRoute('juweliere_checkout_checkout_payment', ["error" => 1]);
        };

        if ($newsletter) {
            try {
                $this->newsletterService->createFromOrder($order);
            } catch (\Throwable) {

            }
        }

        if(!is_null($payone)) { //PAYONE
            try {
                $this->payoneHandler->confirmOrder($order);
            } catch (\Throwable $e) {
                return $this->redirectToRoute("juweliere_checkout_checkout_payment");
            }
        } elseif ($request->getMethod() === "POST") { //PAYPAL
            $payerId = $request->get("payerId");
            if (is_null($payerId)) {
                return $this->redirectToRoute("juweliere_checkout_checkout_payment");
            }

            try {
                $this->paypalHandler->confirmOrder($order, $payerId);
            } catch (\Throwable $e) {
                return $this->redirectToRoute("juweliere_checkout_checkout_payment");
            }
        }


        if (!$this->sessionHandler->getOrderTypeGuest()) {
            try {
                if ($this->sessionHandler->getStoreBillingAddress()) {
                    $userHandler->addAddress($this->sessionHandler->getOrder()->getBillingAddress(), $this->getUser());
                }

                if ($this->sessionHandler->getStoreShippingAddress()) {
                    $userHandler->addAddress($this->sessionHandler->getOrder()->getShippingAddress(), $this->getUser());
                }
            } catch (\Throwable) {
            }
        }

        $this->sessionHandler->unsetOrder();

        $response = $this->render($template, [
            "tracking" => $tracking->getPurchaseData($order),
            "items" => $order->getItems(),
            "user" => $order->getUser(),
            "product" => $order->getItems()[0],
            "order" => $order
        ]);

        $this->decoratorClient->createCheckout($order->getId(), $order->getItems());

        return $response;
    }

    #[Route("/webhook/mail", name: "webhook_mail", options: ["expose" => true])]
    public function webhook(Request $request, Environment $twig, WebhookHandler $webhookHandler): Response
    {
        $disclaimerFile = $this->parameterBag->get('kernel.project_dir') . DIRECTORY_SEPARATOR . "public/pdf/Widerrufsbelehrung_Vogl.pdf";

        $order = json_decode($request->getContent(), true);

        foreach ($order["items"] as $key => $item) {
            $product = $this->productDataProvider->getProductBySku($item["sku"]);
            $item["image"] = $product["images"][0];
            $item["brand"] = $product["brand"];
            $order["items"][$key] = $item;
        }

        try {
            $webhookHandler->mail($order, $disclaimerFile);
        }catch (\Throwable $throwable) {
            var_dump($throwable->getMessage());
            http_response_code(500);
            exit();
        }

        return new JsonResponse(["result" => true]);

    }

    #[Route("/webhook/mail/reminder", name: "webhook_mail_reminder", options: ["expose" => true])]
    public function webhookreminder(Request $request, Environment $twig, WebhookHandler $webhookHandler): Response
    {
        $orders = json_decode($request->getContent(), true);

        try {
            $webhookHandler->remindermail($orders);
        }catch (\Throwable $throwable) {
            var_dump($throwable->getMessage());
            http_response_code(500);
            exit();
        }

        return new JsonResponse(["result" => true]);

    }

    private function getSessionOrderItems(): ?array
    {

        /** @var Order $order */
        $order = $this->sessionHandler->getOrder();

        if (is_null($order)) {
            return null;
        }

        return $order->getItems();
    }

    private function getSessionUserData(): ?UserInterface
    {
        return $this->getUser();
    }
}
