<?php

declare(strict_types=1);

namespace Juweliere\CoreBundle\ElasticSearch;

use Carbon\CarbonImmutable;
use Psr\Cache\InvalidArgumentException;
use Symfony\Component\Cache\Adapter\AdapterInterface;
use Symfony\Component\HttpClient\HttpClient;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;

class ElasticWidgetParser
{
    public const WIDGET_URI = '/api/elastic/widgets';

    private HttpClientInterface $httpClient;
    private AdapterInterface $cache;

    public function __construct(string $baseUri, AdapterInterface $cache)
    {
        $this->httpClient = HttpClient::createForBaseUri($baseUri);
        $this->cache = $cache;
    }

    /**
     * @param string $uuid
     *
     * @return array|null
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     * @throws InvalidArgumentException
     */
    public function getSingleWidget(string $uuid): ?array
    {
        $cachedWidget = $this->cache->getItem(sprintf('widget_%s', $uuid));

        if (!$cachedWidget->isHit()) {
            $uri = sprintf('%s/%s', self::WIDGET_URI, $uuid);
            $response = $this->httpClient->request('GET', $uri);

            if (200 === $response->getStatusCode()) {
                $data = $response->toArray();
                $cachedWidget->expiresAt(CarbonImmutable::now()->addMinute());
                $cachedWidget->set($data);
                $this->cache->save($cachedWidget);
            }
        }

        return $cachedWidget->get();
    }
}
