<?php

declare(strict_types=1);

namespace Juweliere\CoreBundle\Twig;

use Exception;
use Juweliere\CoreBundle\ElasticSearch\ElasticProductsParser;
use Juweliere\CoreBundle\Service\ProductPriceHelper;
use NumberFormatter;
use Symfony\Component\Cache\Adapter\AdapterInterface;
use Twig\Environment;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

class ProductsExtension extends AbstractExtension
{

    const PRODUCT_CDN = 'https://bilder.juweliere.de/cdn';

    private ElasticProductsParser $productsParser;
    private AdapterInterface $cache;
    private ProductPriceHelper $priceHelper;

    public function __construct(ElasticProductsParser $productsParser, ProductPriceHelper $priceHelper, AdapterInterface $cache)
    {
        $this->productsParser = $productsParser;
        $this->cache = $cache;
        $this->priceHelper = $priceHelper;
    }

    /**
     * @return array|TwigFunction[]
     */
    public function getFunctions()
    {
        return [
            'product_image' => new TwigFunction('product_image', [$this, 'getProductImage'], ['is_safe' => ['html']]),
            'product_price' => new TwigFunction('product_price', [$this, 'getProductPrice']),
            'product_zoom' => new TwigFunction('product_zoom', [$this, 'getZoomableImage'], ['needs_environment' => true, 'is_safe' => ['html']]),
        ];
    }

    public function getZoomableImage(Environment $environment, string $image, array $parameters = []): string
    {
        return sprintf('<img id="js-product-zoom" src="%s/2000x2000/%s" class="img-fluid" alt="%s" loading="lazy">', self::PRODUCT_CDN, $image, isset($parameters['title']) ? $parameters['title'] : 'Bild');
    }

    public function getProductPrice(array $product, array $parameters = []): ?string
    {
        try {
            $prices = $this->priceHelper->getPrice($product);
            $numberFormatter = new NumberFormatter('de-DE', NumberFormatter::CURRENCY);

            return $numberFormatter->format($prices['actualPrice']['value']);
        } catch (Exception $e) {
            return null;
        }
    }

    public function getProductImage(string $image, array $parameters = []): string
    {
        $size = isset($parameters['size']) ? $parameters['size'] : 300;

        return sprintf('%s/%sx%s/%s', self::PRODUCT_CDN, $size, $size, $image);
    }
}
