<?php

declare(strict_types=1);

namespace Juweliere\CoreBundle\Controller\Website;

use Exception;
use Juweliere\CoreBundle\ElasticSearch\ElasticProductsParser;
use Juweliere\CoreBundle\Service\ProductPriceHelper;
use Psr\Cache\InvalidArgumentException;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;

class ProductController extends AbstractController
{
    /**
     * @Route("/produkte/{sku}", name="juweliere_catalog_product_show", options={"expose"=true}, requirements={"sku"=".+"}, options={"expose"=true})
     *
     * @param string $sku
     * @param ElasticProductsParser $productsParser
     * @param ProductPriceHelper $priceHelper
     *
     * @return Response
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws InvalidArgumentException
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     * @throws Exception
     */
    public function show(string $sku, ElasticProductsParser $productsParser, ProductPriceHelper $priceHelper): Response
    {
        $product = $productsParser->getSingleProduct($sku);

        if (!$product || $product['isEnabled'] === false || $product['isVisible'] === false) {
            throw $this->createNotFoundException();
        }

        $attributes = [];

        if (isset($product['attributes']) && sizeof($product['attributes'])) {
            foreach ($product['attributes'] as $filter) {
                if ($filter['isFilter']) {
                    $attributes[] = $filter;
                }
            }
        }

        $response = $this->render('pages/product.html.twig', [
            'product' => $product,
            'attributes' => $attributes,
            'price' => $priceHelper->getPrice($product),
        ]);
        $response->setPublic();
        $response->setSharedMaxAge(86400);

        return $response;
    }

    /**
     * @Route("/produkte/{sku}/preview", name="juweliere_catalog_product_preview", options={"expose"=true}, requirements={"sku"=".+"}, options={"expose"=true})
     *
     * @param string $sku
     * @param ElasticProductsParser $productsParser
     * @param ProductPriceHelper $priceHelper
     *
     * @return Response
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws InvalidArgumentException
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     * @throws Exception
     */
    public function preview(string $sku, ElasticProductsParser $productsParser, ProductPriceHelper $priceHelper): Response
    {
        $product = $productsParser->getSingleProduct($sku);
        $response = $this->render('widgets/product-preview.html.twig', [
            'product' => $product,
            'price' => $priceHelper->getPrice($product),
        ]);
        $response->setPublic();
        $response->setSharedMaxAge(86400);

        return $response;
    }

    /**
     * @Route("/products/{sku}/related", name="juweliere_catalog_related_products", options={"expose"=true}, requirements={"sku"=".+"}, options={"expose"=true})
     *
     * @param ElasticProductsParser $productsParser
     * @param string $sku
     *
     * @return Response
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function relatedProducts(ElasticProductsParser $productsParser, string $sku): Response
    {
        $products = $productsParser->getRelatedProducts($sku);

        $response = $this->render(
            'pages/product/related_products.html.twig',
            ['products' => $products]
        );

        $response->setPublic();
        $response->setSharedMaxAge(86400);

        return $response;
    }
}
