<?php

declare(strict_types=1);

namespace Juweliere\CoreBundle\Controller\Website;

use Juweliere\CoreBundle\ElasticSearch\ElasticProductsParser;
use Juweliere\CoreBundle\Session\CatalogFilterManager;
use Exception;
use Sulu\Component\Content\Compat\StructureInterface;
use Sulu\Component\DocumentManager\DocumentManagerInterface;
use Sulu\Component\DocumentManager\Exception\DocumentManagerException;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;

class SelectionController extends WebsiteController
{
    private ElasticProductsParser $productsParser;
    private CatalogFilterManager $filterManager;
    private DocumentManagerInterface $documentManager;

    public function __construct(DocumentManagerInterface $documentManager, ElasticProductsParser $productsParser, CatalogFilterManager $filterManager)
    {
        $this->productsParser = $productsParser;
        $this->filterManager = $filterManager;
        $this->documentManager = $documentManager;
    }

    /**
     * @param Request $request
     * @param StructureInterface $structure
     * @param bool $preview
     * @param bool $partial
     *
     * @return Response
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     * @throws Exception
     */
    public function index(Request $request, StructureInterface $structure, $preview = false, $partial = false): Response
    {
        $this->updateFilters($request, $structure);

        $queryId = $structure->getPropertyValue('selection_uuid');

        if ($queryId) {
            return $this->renderStructure(
                $request,
                $structure,
                [
                    'selection_uuid' => $structure->getPropertyValue('selection_uuid'),
                    'result' => $this->productsParser->getProductQuery($request, $queryId),
                    'query' => $queryId,
                ],
                $preview,
                $partial
            );
        }

        return $this->renderStructure(
            $request,
            $structure,
            [],
            $preview,
            $partial
        );
    }

    /**
     * @Route("/api/selections/{uuid}/grid", name="juweliere_catalog_selection_grid", options={"expose"=true})
     *
     * @param Request $request
     * @param string $uuid
     *
     * @return Response
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function selectionsGrid(Request $request, string $uuid): Response
    {
        $page = $request->query->get('page');

        if ($page) {
            $this->filterManager->setPage($uuid, (int) $page);
        }

        return $this->render('pages/catalog/_grid.html.twig', [
            'result' => $this->productsParser->getProductQuery($request, $uuid),
            'uuid' => $uuid,
        ]);
    }

    /**
     * @Route("/api/session/catalog/{uuid}", name="juweliere_catalog_session_add_product_filter")
     *
     * @param Request $request
     * @param string $uuid
     *
     * @return JsonResponse
     */
    public function addFilterToSession(Request $request, string $uuid): JsonResponse
    {
        $filters = $request->query->get('filters');

        if ($filters) {
            foreach ($filters as $key => $filter) {
                $data = [
                    'code' => $key,
                    'value' => $filter,
                ];
                $this->filterManager->addFilter($uuid, $data);
            }

            $this->filterManager->setPage($uuid, 1);
        }

        return new JsonResponse($this->filterManager->getEntry($uuid), 204);
    }

    /**
     * @Route("/api/session/catalog/{uuid}/delete/all", name="juweliere_catalog_session_delete_all_product_filters", methods={"DELETE"})
     *
     * @param string $uuid
     *
     * @return JsonResponse
     */
    public function deleteFiltersFromSession(string $uuid): JsonResponse
    {
        $this->filterManager->removeEntry($uuid);

        return new JsonResponse(null, 204);
    }

    /**
     * @Route("/api/session/catalog/{uuid}/delete/{filter}", name="juweliere_catalog_session_delete_product_filter")
     *
     * @param string $uuid
     * @param string $filter
     *
     * @return RedirectResponse
     * @throws DocumentManagerException
     */
    public function deleteFilterFromSession(string $uuid, string $filter): RedirectResponse
    {
        $this->filterManager->removeFilter($uuid, $filter);

        $document = $this->documentManager->find($uuid);

        return new RedirectResponse($document->getResourceSegment());
    }

    /**
     * @param Request $request
     * @param StructureInterface $structure
     */
    private function updateFilters(Request $request, StructureInterface $structure): void
    {
        $uuid = (string) $structure->getUuid();
        $filters = $request->query->get('filters');
        $page = $request->query->get('page');
        $limit = $request->query->get('limit');
        $sort = $request->query->get('sort');
        $dir = $request->query->get('dir', 'asc');

        if ($filters) {
            foreach ($filters as $key => $filter) {
                $data = [
                    'code' => $key,
                    'value' => $filter,
                ];
                $this->filterManager->addFilter($uuid, $data);
            }
        }

        if ($page) {
            $this->filterManager->setPage($uuid, (int) $page);
        }

        if ($limit) {
            $this->filterManager->setLimit($uuid, (int) $limit);
        }

        if ($sort) {
            $this->filterManager->setSorting($uuid, $sort, $dir);
        }
    }
}
