<?php

declare(strict_types=1);

namespace Juweliere\CoreBundle\Controller\Website;

use Psr\Cache\InvalidArgumentException;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Cache\Adapter\AdapterInterface;
use Symfony\Component\HttpClient\HttpClient;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;

class FilterController extends AbstractController
{
    private string $baseUri;

    private AdapterInterface $cache;

    public function __construct(string $baseUri, AdapterInterface $cache)
    {
        $this->baseUri = $baseUri;
        $this->cache = $cache;
    }

    /**
     * @Route("/attributes/{code}/name", name="juweliere_catalog_attribute_name")
     *
     * @param Request $request
     * @param string $code
     *
     * @return Response
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws InvalidArgumentException
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function attribute(Request $request, string $code): Response
    {
        $uuid = $request->get('uuid');
        $cachedAttribute = $this->cache->getItem('filter_'.$code);
        $attribute = null;

        if ($cachedAttribute->isHit()) {
            $attribute = $cachedAttribute->get();

            $response = $this->render(
                'pages/product/filter.html.twig',
                [
                    'attribute' => $attribute,
                    'options' => $this->bucketLabels($attribute, $request->get('buckets')),
                    'uuid' => $uuid,
                ]
            );

            $response->setPublic();
            $response->setMaxAge(60);

            return $response;
        }

        $client = HttpClient::create();
        $attributeResponse = $client->request('GET', sprintf('/api/elastic/attributes/%s', $code), [
            'base_uri' => $this->baseUri,
        ]);

        if ($attributeResponse->getStatusCode() < 400) {
            $attribute = $attributeResponse->toArray(false);
            $cachedAttribute->set($attribute);
            $this->cache->save($cachedAttribute);
        }

        $response = $this->render(
            'pages/product/filter.html.twig',
            [
                'attribute' => $attribute,
                'options' => $this->bucketLabels($attribute, $request->get('buckets')),
                'uuid' => $uuid,
            ]
        );

        $response->setPublic();
        $response->setMaxAge(60);

        return $response;
    }

    /**
     * @Route("/filters/{uuid}", name="juweliere_catalog_session_filters")
     *
     * @param SessionInterface $session
     * @param string $uuid
     *
     * @return Response
     *
     * @throws InvalidArgumentException
     */
    public function filters(SessionInterface $session, string $uuid): Response
    {
        if ($session->has($uuid)) {

            $data = [];
            foreach ($session->get($uuid)['filters'] as $key => $filter) {

                $cachedAttribute = $this->cache->getItem('filter_'.$key);
                $attribute = null;

                if ($cachedAttribute->isHit()) {
                    $attribute = $cachedAttribute->get();
                }

                if ($attribute) {
                    $label = null;
                    foreach ($attribute['options'] as $option) {

                        if ($option['code'] === $filter) {
                            $label = $option['label'];
                        }
                    }

                    $data[] = [
                        'attribute' => [
                            'key' => $key,
                            'value' => $attribute['name'],
                        ],
                        'option' => [
                            'key' => $filter,
                            'value' => $label,
                        ],
                    ];
                }
            }

            $response = $this->render(
                'pages/product/selection.html.twig',
                [
                    'selectedFilters' => $data,
                    'uuid' => $uuid,
                ]
            );


            $response->setPublic();
            $response->setMaxAge(60);

            return $response;
        }

        return new Response();
    }

    /**
     * @param $attribute
     * @param array $buckets
     *
     * @return array
     *
     * @throws InvalidArgumentException
     */
    protected function bucketLabels($attribute, array $buckets): array
    {
        $options = [];

        foreach ($buckets as $bucket) {
            $cachedOption = $this->cache->getItem('option_'.$bucket['key']);

            if ($cachedOption->isHit()) {
                $options[] = [
                    'key' => $bucket['key'],
                    'label' => $cachedOption->get(),
                ];
            } else {
                foreach ($attribute['options'] as $option) {
                    if ($option['code'] === $bucket['key']) {
                        $options[] = [
                            'key' => $bucket['key'],
                            'label' => $option['label'],
                        ];

                        $cachedOption->set($option['label']);
                        $this->cache->save($cachedOption);
                    }
                }
            }
        }

        return $options;
    }
}
