<?php

declare(strict_types=1);

namespace Juweliere\CoreBundle\ElasticSearch;

use Juweliere\CoreBundle\Session\CatalogFilterManager;
use Psr\Cache\InvalidArgumentException;
use Symfony\Component\Cache\Adapter\AdapterInterface;
use Symfony\Component\HttpClient\HttpClient;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;

class ElasticProductsParser
{

    public const PRODUCTS_URI = '/api/elastic/products';
    public const SEARCH_URI = '/api/elastic/search/products';
    public const RELATED_PRODUCTS_URI = '/api/elastic/related-products';
    public const QUERY_URI = 'api/elastic/queries';

    private SessionInterface $session;
    private CatalogFilterManager $catalogFilter;
    private AdapterInterface $cache;
    private HttpClientInterface $httpClient;

    public function __construct(string $baseUri, SessionInterface $session, CatalogFilterManager $catalogFilter, AdapterInterface $cache)
    {
        $this->session = $session;
        $this->catalogFilter = $catalogFilter;
        $this->cache = $cache;
        $this->httpClient = HttpClient::createForBaseUri($baseUri);
    }

    /**
     * @param string $sku
     * @param bool $cache
     *
     * @return array|null
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws InvalidArgumentException
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function getSingleProduct(string $sku, bool $cache = true): ?array
    {
        $cachedProduct = $this->cache->getItem(sprintf('product_%s', $sku));

        if (!$cache || !$cachedProduct->isHit()) {
            $uri = sprintf('%s/%s', self::PRODUCTS_URI, $sku);
            $response = $this->httpClient->request('GET', $uri);

            if (200 === $response->getStatusCode()) {
                $data = $response->toArray();
                $cachedProduct->set($data);
                $this->cache->save($cachedProduct);
            }
        }

        return $cachedProduct->get();
    }

    /**
     * @param string $sku
     * @return array|null
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function getRelatedProducts(string $sku): ?array
    {
        $uri = sprintf('%s/%s?random=1&onhand=0', self::RELATED_PRODUCTS_URI, $sku);

        $response = $this->httpClient->request('GET', $uri);

        if (200 === $response->getStatusCode()) {
            return $response->toArray();
        }

        return null;
    }

    /**
     * @param Request $request
     * @param string  $id
     *
     * @return array|null
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function getProductQuery(Request $request, string $id): ?array
    {
        $uuid = $request->attributes->get('uuid');
        if ($request->attributes->has('structure')) {
            $uuid = $request->attributes->get('structure')->getUuid();
        }

        $params = $request->query->all();
        $sessionFilters = $this->catalogFilter->getEntry($uuid);

        if ($sessionFilters) {
            $params = array_merge($sessionFilters, $request->query->all());
        }

        $uri = sprintf('%s/%s?%s', self::QUERY_URI, $id, http_build_query($params));
        $response = $this->httpClient->request('GET', $uri);

        if (200 === $response->getStatusCode()) {
            return $response->toArray();
        }

        return null;
    }

    /**
     * @param Request|null $request
     *
     * @param int|null $limit
     * @return array|null
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function searchProducts(?Request $request = null, ?int $limit = 128): ?array
    {
        $uri = sprintf('%s?limit=%s', self::SEARCH_URI, $limit);

        if ($request) {
            $q = $request->query->get('q');
            $limit = $request->query->get('limit', 128);

            $uri = sprintf('%s?q=%s&limit=%s', self::SEARCH_URI, $q, $limit);
        }

        $response = $this->httpClient->request('GET', $uri);

        if (200 === $response->getStatusCode()) {
            return $response->toArray();
        }

        return null;
    }
}
