<?php

declare(strict_types=1);

namespace Juweliere\CoreBundle\Twig;

use Juweliere\CoreBundle\ElasticSearch\ElasticWidgetParser;
use Juweliere\CoreBundle\ElasticSearch\ElasticProductsParser;
use Psr\Cache\InvalidArgumentException;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Twig\Environment;
use Twig\Error\LoaderError;
use Twig\Error\RuntimeError;
use Twig\Error\SyntaxError;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

class WidgetExtension extends AbstractExtension
{
    private ElasticWidgetParser $widgetParser;
    private ElasticProductsParser $productsParser;

    public function __construct(ElasticWidgetParser $widgetParser, ElasticProductsParser $productsParser)
    {
        $this->widgetParser = $widgetParser;
        $this->productsParser = $productsParser;
    }

    /**
     * @return array
     */
    public function getFunctions(): array
    {
        return [
            'widget' => new TwigFunction('widget', [$this, 'getWidget'], ['needs_environment' => true, 'is_safe' => ['html']]),
        ];
    }

    /**
     * @param Environment $environment
     * @param $uuid
     * @param array $parameters
     *
     * @return string|null
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     * @throws LoaderError
     * @throws RuntimeError
     * @throws SyntaxError|InvalidArgumentException
     */
    public function getWidget(Environment $environment, $uuid, $parameters = []): ?string
    {
        $item = $this->widgetParser->getSingleWidget($uuid);
        $type = isset($parameters['type']) ? $parameters['type'] : $item['type']['slug'];

        if ($item) {
            switch ($type) {
                case '3_products':
                    return $environment->render('widgets/three-products.html.twig', [
                        'widget' => $item,
                        'type' => $type,
                        'products' => $this->getProducts($item),
                    ]);

                case '4_products':
                    return $environment->render('widgets/four-products.html.twig', [
                        'widget' => $item,
                        'type' => $type,
                        'products' => $this->getProducts($item),
                    ]);

                case '6_products':
                    return $environment->render('widgets/six-products.html.twig', [
                        'widget' => $item,
                        'type' => $type,
                        'products' => $this->getProducts($item),
                    ]);

                case '8_products':
                    return $environment->render('widgets/eight-products.html.twig', [
                        'widget' => $item,
                        'type' => $type,
                        'products' => $this->getProducts($item),
                    ]);

                case 'product_carousel':
                    return $environment->render('widgets/product-carousel.html.twig', [
                        'widget' => $item,
                        'type' => $type,
                        'class' => isset($parameters['class']) ? $parameters['class'] : 'products-carousel',
                        'products' => $this->getProducts($item),
                    ]);
            }
        }

        return null;
    }

    /**
     * @param array $widget
     * @return array
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws InvalidArgumentException
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    private function getProducts(array $widget): array
    {
        $products = [];

        foreach ($widget['items'] as $item) {
            $product = $this->productsParser->getSingleProduct($item['product']['sku']);

            if ($product) {
                $products[] = $product;
            }
        }

        return $products;
    }
}
