<?php


namespace Juweliere\JuwApiClient\Service;


use Juweliere\JuwApiClient\Constants;
use League\OAuth2\Client\Provider\GenericProvider;
use League\OAuth2\Client\Token\AccessToken;
use League\OAuth2\Client\Token\AccessTokenInterface;
use Predis\Client;
use Psr\Http\Message\RequestInterface;
use Psr\Log\LoggerInterface;

trait ServiceTrait
{
    private GenericProvider $provider;
    protected $baseUrl;
    private Client $redis;
    private LoggerInterface $logger;

    /**
     * ServiceTrait constructor.
     * @param GenericProvider $provider
     * @param $baseUrl
     * @param Client $redis
     * @param LoggerInterface $logger
     */
    public function __construct(GenericProvider $provider, $baseUrl, Client $redis, LoggerInterface $logger)
    {
        $this->provider = $provider;
        $this->baseUrl = $baseUrl . $this->baseUrlResource;
        $this->redis = $redis;
        $this->logger = $logger;
    }

    /**
     * @return AccessToken|AccessTokenInterface
     */
    protected function getAccessToken($force = false)
    {
        $accessToken = $this->redis->get(Constants::KEY_ACCESS_TOKEN);

        if (!$accessToken || $force) {
            try {
                $this->logger->debug("Get new access token");

                $accessToken = $this->provider->getAccessToken('client_credentials');

                $this->redis->set(Constants::KEY_ACCESS_TOKEN, serialize($accessToken));
                $this->redis->expire(Constants::KEY_ACCESS_TOKEN, $accessToken->getExpires() - time());
            } catch (\Throwable $e) {
                $this->logger->error($e->getMessage());
                throw $e;
            }
        } else {
            $accessToken = unserialize($accessToken);
            $this->logger->debug("Use cached token");
        }

        return $accessToken;
    }

    /**
     * @param RequestInterface $request
     * @param $targetObject
     * @return mixed
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    protected function getResponse(RequestInterface $request, $targetObject, $isPaginated = false)
    {
        $httpResult = $this->provider->getHttpClient()->send($request);

        if ($isPaginated) {
            $hydraData = json_decode((string)$httpResult->getBody());
            $data = $hydraData->{"hydra:member"};
        } else {
            $data = json_decode((string)$httpResult->getBody());
        }

        $mapper = (new \JsonMapper\JsonMapperFactory())->default();

        try {
            if (is_array($data)) {
                $result = [];
                foreach ($data as $object) {
                    $targetObjectNew = clone $targetObject;
                    $mapper->mapObject($object, $targetObjectNew);
                    $result[] = $targetObjectNew;
                }
            } else {
                $mapper->mapObject($data, $targetObject);
                $result = $targetObject;
            }
        } catch (\Throwable $e) {
            $this->logger->error($e->getMessage());
            throw $e;
        }

        if ($isPaginated) {
            $resultPaginated =
                [
                    "items" => $result,
                    "totalItems" => $hydraData->{"hydra:totalItems"},
                    "pagination" => $this->getPagination($hydraData->{"hydra:view"})
                ];

            return $resultPaginated;
        }

        return $result;
    }


    protected function getPagination($data) {

        $pagination = [];

        $mapping = [
            "@id" => "current",
            "hydra:first" => "first",
            "hydra:last" => "last",
            "hydra:previous" => "previous",
            "hydra:next" => "next"
        ];

        foreach ($mapping as $hydra => $key) {
            if (isset($data->$hydra)) {
                $pagination[$key] = intval(substr($data->$hydra, strrpos($data->{$hydra}, '=') + 1));
            }
        }

        return $pagination;
    }
}
