<?php


namespace Juweliere\JuwApiClient;

use Juweliere\JuwApiClient\Service\Addresses;
use Juweliere\JuwApiClient\Service\Me;
use Juweliere\JuwApiClient\Service\Orders;
use Juweliere\JuwApiClient\Service\ServiceTrait;
use Juweliere\JuwApiClient\Service\Users;
use Juweliere\JuwApiClient\Service\WishlistItems;
use League\OAuth2\Client\Provider\GenericProvider;
use Monolog\Formatter\LineFormatter;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Logger;
use Predis\Client;
use Psr\Log\LoggerInterface;

class JuwApiClient
{
    const PAYMENT_TYPE_STRIPE = "STRIPE";
    const PAYMENT_TYPE_STRIPE_CS = "STRIPE_CS";
    const PAYMENT_TYPE_PAYPAL = "PAYPAL";
    const PAYMENT_TYPE_MOLLIE = "MOLLIE";
    const PAYMENT_TYPE_ADYEN = "ADYEN";

    public Orders $orders;
    public Me $me;
    public Users $users;
    public WishlistItems $wishlistItems;
    public Addresses $addresses;

    private LoggerInterface $logger;
    private Client $redis;
    private GenericProvider $provider;
    private string $baseUrl;

    /**
     * JuwApiClient constructor.
     * @param string $baseUrl
     * @param string $clientId
     * @param string $clientSecret
     * @param string $logPath
     */
    public function __construct(
        $baseUrl,
        $clientId,
        $clientSecret,
        $logPath,
        $redisHost,
        $redisPort,
        $redisPass,
        $redisDb,
        $redisPrefix,
        $logLevel = Logger::DEBUG
    )
    {
        $this->initRedis($redisHost, $redisPort, $redisPass, $redisDb, $redisPrefix);
        $this->initLogger($logPath, $logLevel);
        $this->initProvider($clientId, $clientSecret, $baseUrl);
        $this->initServices($baseUrl);
    }

    /**
     * @param GenericProvider $provider
     * @param $baseUrl
     */
    private function initServices($baseUrl)
    {
        $this->orders = new Orders($this->provider, $baseUrl, $this->redis, $this->logger);
        $this->me = new Me($this->provider, $baseUrl, $this->redis, $this->logger);
        $this->users = new Users($this->provider, $baseUrl, $this->redis, $this->logger);
        $this->wishlistItems = new WishlistItems($this->provider, $baseUrl, $this->redis, $this->logger);
        $this->addresses = new Addresses($this->provider, $baseUrl, $this->redis, $this->logger);
    }

    private function initProvider($clientId, $clientSecret, $baseUrl)
    {
        $this->provider = new GenericProvider([
            'clientId' => $clientId,
            'clientSecret' => $clientSecret,
            'urlAccessToken' => $baseUrl . "/token",
            'urlAuthorize' => "",
            'urlResourceOwnerDetails' => ""
        ]);

        $httpClientProvider = new HttpClientProvider($this->logger);

        $this->provider->setHttpClient($httpClientProvider->get());
    }

    private function initLogger($logPath, $logLevel)
    {
        /** @var Logger $logger */
        $logger = new Logger("juw_api");

        /** @var RotatingFileHandler $fileHandler */
        $fileHandler = new RotatingFileHandler(
            $logPath . "juw_api.log",
            10
        );

        $fileHandler->setLevel($logLevel);

        $fileHandler->setFormatter(new LineFormatter(null, null, true));

        $logger->pushHandler($fileHandler);

        $this->logger = $logger;
    }

    private function initRedis(
        $redisHost,
        $redisPort,
        $redisPass,
        $redisDb,
        $redisPrefix
    )
    {
        $redisConnection =  [
            'scheme'    => 'tcp',
            'host'      => $redisHost,
            'port'      => $redisPort,
            'database'  => $redisDb
        ];

        $redisOptions = [
            'prefix'    => $redisPrefix
        ];

        if (trim($redisPass) !== "") {
            $redisConnection['password'] = $redisPass;
        }

        $this->redis = new Client($redisConnection, $redisOptions);
    }


}
