<?php

declare(strict_types=1);

namespace Juweliere\ProductBundle\Controller;

use Juweliere\ProductBundle\ElasticSearch\ElasticProductsParser;
use Juweliere\ProductBundle\Service\ProductDataHelper;
use Sulu\Bundle\ContactBundle\Entity\Contact;
use Sulu\Bundle\HttpCacheBundle\Cache\SuluHttpCache;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Cache\Adapter\AdapterInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Component\Routing\Annotation\Route;

/**
 * @Route("/products", name="juweliere_products_products_")
 */

class ProductController extends AbstractController
{
    const PRODUCT_TYPE_WATCH = "uhren";
    const PRODUCT_TYPE_JEWELLERY = "schmuck";
    const SESSION_LAST_SEEN = "lastseen";

    /**
     * @Route("/{sku}", name="show", requirements={"sku"=".+"}, options={"expose"=true})
     *
     * @param Request $request
     * @param string $sku
     * @param ElasticProductsParser $productsParser
     * @param ProductDataHelper $productDataHelper
     * @param AdapterInterface $cacheAdapter
     * @param SessionInterface $session
     *
     * @return Response
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function show(
        Request               $request,
        string                $sku,
        ElasticProductsParser $productsParser,
        ProductDataHelper     $productDataHelper,
        AdapterInterface      $cacheAdapter,
        SessionInterface      $session): Response
    {
	/*
        $cachedProduct = $cacheAdapter->getItem(sprintf('product_%s', $sku));

        if (!$cachedProduct->isHit()) {
            $product = $productsParser->getSingleProduct($sku);
            $cachedProduct->set($product);
            $cacheAdapter->save($cachedProduct);
        }

        $product = $cachedProduct->get();
	*/
	    $product = $productsParser->getSingleProduct($sku);

	    if (is_null($product) || !$product["isEnabled"]) {
            return $this->renderError();
        }

	    $product = $productDataHelper->consolidateGalleryImages($product);

        $attributes = [];

        if (isset($product['attributes']) && sizeof($product['attributes'])) {
            foreach ($product['attributes'] as $filter) {
//                if ($this->container->hasParameter('attributes')) {
//                    $blacklistAttributes = $this->getParameter('attributes')['blacklist'];
//                    if (!in_array($filter['attributeCode'], $blacklistAttributes) && $filter['isFilter']) {
//                        $attributes[] = $filter;
//                    }
//                }

                if ($filter['isFilter']) {
                    $attributes[] = $filter;
                }
            }
        }

        if (isset($product['attributes']) && sizeof($product['attributes'])) {
            foreach ($product['attributes'] as $filter) {
//                if ($this->container->hasParameter('attributes')) {
//                    $blacklistAttributes = $this->getParameter('attributes')['blacklist'];
//                    if (!in_array($filter['attributeCode'], $blacklistAttributes) && $filter['isFilter']) {
//                        $attributes[] = $filter;
//                    }
//                }

                if ($filter['isFilter']) {
                    $attributes[] = $filter;
                }
            }
        }

        if (isset($product['description']) && !empty($product['description'])) {

            if (preg_match("/^<div/", $product["description"])) {
                $count = substr_count($product['description'], '<div>');

                $product['description'] = str_replace('<span style="font-weight: bold;">', '<strong>', $product['description'] );
                $product['description'] = str_replace('</span>', '</strong><br/>', $product['description'] );

                $product['description'] = str_replace('<div><span', '<span', $product['description'] );
                $product['description'] = str_replace('<br/></div>', '<br/>', $product['description'] );
                $product['description'] = str_replace('<br>', '', $product['description'] );
                $product['description'] = str_replace('<div>', '', $product['description'] );
                $product['description'] = str_replace('</strong><br/></div>', '</strong><br/>', $product['description'] );
                $product['description'] = str_replace('</div>', '<br/>', $product['description'] );
                $product['description'] = str_replace('<br/><br/>', '<br/>', $product['description'] );

                $count = substr_count($product['description'], '<strong>');

                $maxItemsPerCol = (int)round(($count) / 2);

                $count = 0;

                $description = '<div class="col-md-6">';
                foreach(explode("<strong>", $product['description']) as $line){
                    if (!empty(trim($line))) {
                        $description .= '<p><strong>' . $line . "</p>";
                        $count++;
                        if ($count == $maxItemsPerCol) {
                            $description .= '</div><div class="col-md-6">';
                            #$count = 0;
                        }
                    }

                }

                $description .= '</div>';
            } else {
                $product['description'] = str_replace('<p style="text-align: left;">', '', $product['description'] );
                $product['description'] = str_replace('</p>', '', $product['description'] );
                $product['description'] = str_replace('<p>', '', $product['description'] );
                $product['description'] = str_replace('<strong><br />', '<strong>', $product['description'] );

                // /products/tudor-m12313-0003
                if (preg_match("/span style=\"font-weight: bold;\"/", $product['description'])) {
                    $product['description'] = str_replace('span style="font-weight: bold;"', 'strong', $product['description'] );
                    $product['description'] = str_replace('<span style="font-weight: 700;"><br></span>', '<br>', $product['description'] );

                    $product['description'] = str_replace('/span', '/strong', $product['description'] );
                }

                $count = substr_count($product['description'], '<strong>');

                $maxItemsPerCol = (int)round(($count) / 2);

                $count = 0;

                $description = '<div class="col-md-6">';
                foreach(explode("<strong>", $product['description']) as $line){
                    if (!empty(trim($line))) {
                        $description .= '<p><strong>' . nl2br($line) . "</p>";
                        $count++;
                        if ($count == $maxItemsPerCol) {
                            $description .= '</div><div class="col-md-6">';
                            #$count = 0;
                        }
                    }

                }

                $description .= '</div>';
            }

            $product["description"] = $description;
        }

        $device = $request->get('device');

        switch ($device) {
            case 'mobile':
                $template = 'pages/product/mobile/_product.html.twig';
                break;
            case 'tablet':
                $template = 'pages/product/tablet/_product.html.twig';
                break;
            default:
                $template = 'pages/product/desktop/_product.html.twig';
                break;
        }

        $response = $this->render($template, [
            'product' => $product,
            'attributes' => $attributes,
            'price' => is_array($product) ? $productDataHelper->getPrice($product) : null,
            'contact_person' => $this->getContacData($product),
            'lastseen_products' => $this->getLastSeenProducts($product, $session)
        ]);
        $response->setSharedMaxAge(86400);
        $response->headers->addCacheControlDirective('must-revalidate', true);

        return $response;
    }

    /**
     * @Route("/products/{sku}/related", name="related_products", options={"expose"=true}, requirements={"sku"=".+"}, options={"expose"=true})
     *
     * @param ElasticProductsParser $productsParser
     * @param Request $request
     * @param string $sku
     * @param bool $brand
     * @param bool $small
     * @return Response
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function relatedProducts(ElasticProductsParser $productsParser, Request $request, string $sku, bool $brand = false, bool $small = false)
    {
        $products = $productsParser->getRelatedProducts($sku, true, $brand);

        $device = $request->get('device');

        switch ($device) {
            case 'tablet':
                $template = 'pages/product/tablet/_related';
                break;
            case 'mobile':
                $template = 'pages/product/mobile/_related';
                break;

            case 'desktop':
            default:
                $template = 'pages/product/desktop/_related';
                break;
        }

        if ($small) {
            $template .= "_small";
        }
        $template .= ".html.twig";

        $productGroup = "unknown";
        if (isset($products[0])) {
            $productGroup = $this->getProductType($products[0]);
        }

        $response = $this->render(
            $template,
            [
                'products' => $products,
                'product_group' => $productGroup
            ]
        );

        $response->setMaxAge(0);
        $response->setSharedMaxAge(0);
        $response->headers->set('Cache-Control', 'no-cache, must-revalidate');
        $response->headers->set(
            SuluHttpCache::HEADER_REVERSE_PROXY_TTL,
            $response->getAge()
        );

        return $response;
    }

    /**
     * @param $product
     * @return Contact|null
     */
    private function getContacData($product)
    {
        $contactsRepo = $this->getDoctrine()
            ->getRepository(Contact::class);

        /** @var string|null $brand */
        $brand = $this->getProductBrand($product);

        /** @var string $productType */
        $productType = $this->getProductType($product);

        /** @var string|null $contact */
        $contact = null;

        if ($brand) {
            $contact = $contactsRepo
                ->createQueryBuilder("c")
                ->leftJoin('c.tags', 't')
                ->where("t.name = '" . $brand . "'")
                ->getQuery()
                ->getOneOrNullResult();
        }

        if(is_null($contact)) {
            $contact = $contactsRepo
                ->createQueryBuilder("c")
                ->leftJoin('c.tags', 't')
                ->where("t.name = '" . $productType . "'")
                ->getQuery()
                ->getOneOrNullResult();
        }

        return $contact;
    }

    /**
     * @param $product
     * @return string
     */
    private function getProductBrand($product)
    {
        if (!empty($product['attributes'])) {
            foreach ($product["attributes"] as $attribute) {
                if ($attribute["attributeCode"] == "juweliere_brand") {
                    return $attribute["valueCode"];
                }
            }
        }

        return null;
    }

    /**
     * @param $product
     * @return string
     */
    private function getProductType($product)
    {
        if (!empty($product['attributes'])) {
            foreach ($product["attributes"] as $attribute) {
                if ($attribute["attributeCode"] == "schmuckart") {
                    return self::PRODUCT_TYPE_JEWELLERY;
                }
            }
        }

        return self::PRODUCT_TYPE_WATCH;
    }

    /**
     * @param $product
     * @param SessionInterface $session
     * @return array|mixed
     */
    private function getLastSeenProducts($product, SessionInterface $session)
    {
        /** @var array $lastSeenProducts */
        $lastSeenProducts = array();

        /** @var array $newLastSeenProducts */
        $newLastSeenProducts = array();

        /** @var int $count */
        $count = 0;

        /** @var array $attributes */
        $attributes = array(
            "sku",
            "image",
            "brand",
            "name",
            "showPrice",
            "rrp"
        );

        if ($session->has(self::SESSION_LAST_SEEN)) {
            $lastSeenProducts = $session->get(self::SESSION_LAST_SEEN);
        }

        if (is_null($product)) {
            return $lastSeenProducts;
        }

        foreach ($lastSeenProducts as $key => $lastSeenProduct) {
            if ($count > 4) {
                break;
            }
            if ($lastSeenProduct["sku"] === $product["sku"]) {
                continue;
            }
            $newLastSeenProducts[] = $lastSeenProduct;
            $count++;
        }

        /** @var array $productDataFiltered */
        $productDataFiltered = array_intersect_key($product, array_flip($attributes));

        array_unshift($newLastSeenProducts, $productDataFiltered);

        $session->set(self::SESSION_LAST_SEEN, $newLastSeenProducts);
        $session->save();

        array_shift($newLastSeenProducts);

        return $newLastSeenProducts;
    }

    public function renderError()
    {
        $content = $this->renderView(
            "@JuweliereProduct/error404.html.twig");
        $response = new Response($content, 404);

        return $response;
    }
}
