<?php

declare(strict_types=1);

namespace Juweliere\ProductBundle\Decorator;

use Monolog\Formatter\LineFormatter;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Logger;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpClient\ScopingHttpClient;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;

abstract class AbstractDecoratorApiClient
{
    protected $httpClient;
    private $baseUrl;
    private $accessToken;
    protected LoggerInterface $logger;

    /**
     * AbstractElasticParser constructor.
     * @param HttpClientInterface $httpClient
     * @param $baseUrl
     * @param $accessToken
     */
    public function __construct(
        HttpClientInterface $httpClient, $baseUrl, $accessToken,
                            $logPath,
                            $logLevel)
    {
        $this->baseUrl = $baseUrl;
        $this->accessToken = $accessToken;
        $this->initLogger($logPath, $logLevel);

        $this->httpClient = new ScopingHttpClient($httpClient, [
            'base_uri' => $this->baseUrl,
            'http_version' => '1.1'
        ]);

        $this->httpClient->setLogger($this->logger);
    }

    /**
     * @param string $uuid
     *
     * @return array|null
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function get(string $uri, $auth = true): ?array
    {
        $response = $this->httpClient->request('GET', $uri, [
            'base_uri' => $this->baseUrl,
            'http_version' => '1.1',
            'headers' => [
                "X-Auth-ApiKey" => $this->accessToken
            ]
        ]);

        if (200 === $response->getStatusCode()) {
            return $response->toArray();
        }

        return null;
    }

    /**
     * @param string $uuid
     *
     * @return array|null
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function post(string $uri, $data): ?array
    {
        $response = $this->httpClient->request('POST', $uri, [
            'base_uri' => $this->baseUrl,
            'http_version' => '1.1',
            'headers' => [
                "X-Auth-ApiKey" => $this->accessToken
            ],
            'body' => json_encode($data)
        ]);

        $this->logger->info(json_encode($response->getInfo("debug")));

        if (200 === $response->getStatusCode()) {
            return $response->toArray();
        }

        return null;
    }

    private function initLogger($logPath, $logLevel)
    {
        /** @var Logger $logger */
        $logger = new Logger("juw_decorator");

        /** @var RotatingFileHandler $fileHandler */
        $fileHandler = new RotatingFileHandler(
            $logPath . "juw_decorator.log",
            10
        );

        $fileHandler->setLevel($logLevel);

        $fileHandler->setFormatter(new LineFormatter(null, null, true));

        $logger->pushHandler($fileHandler);

        $this->logger = $logger;
    }
}
