<?php

namespace Juweliere\ProductBundle\ElasticSearch;

use Juweliere\ProductBundle\Session\CatalogFilterManager;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;
use Juweliere\ProductBundle\Decorator\AbstractDecoratorApiClient;

class ElasticProductsParser extends AbstractDecoratorApiClient
{
    private $catalogFilter;

    public function __construct(
        HttpClientInterface $httpClient,
        CatalogFilterManager $catalogFilter,
        $baseUrl,
        $accessToken,
        $logPath,
        $logLevel)
    {
        parent::__construct($httpClient, $baseUrl, $accessToken, $logPath, $logLevel);
        $this->catalogFilter = $catalogFilter;
    }

    /**
     * @param string $sku
     *
     * @return array|null
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function getSingleProduct(string $sku): ?array
    {
        $uri = sprintf('/api/elastic/products/%s', $sku);
        return $this->get($uri);
    }

    /**
     * @param string $sku
     *
     * @return array|null
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function search(string $query): ?array
    {
        $uri = sprintf('/api/elastic/search/products?q=%s', $query);
        return $this->get($uri);
    }

    /**
     * @param string $sku
     * @param bool $random
     *
     * @param bool $brandProducts
     * @return array|null
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function getRelatedProducts(string $sku, bool $random = false, bool $brandProducts = false): ?array
    {
        $params = array(
            "random" => 0,
            "brand" => 0
        );

        if ($random) {
            $params["random"] = 1;
        }

        if ($brandProducts) {
            $params["brand"] = 1;
        }

        $uri = sprintf('/api/elastic/related-products/%s?%s', $sku, http_build_query($params));
        return $this->get($uri);
    }

    /**
     * @param Request $request
     * @param string  $id
     *
     * @return array|null
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function getProductQuery(Request $request, string $id): ?array
    {
        if ($request->attributes->has('structure')) {
            $uuid = $request->attributes->get('structure')->getUuid();
        } else {
            $uuid = $request->attributes->get('uuid');
        }

        $sessionFilters = $this->catalogFilter->getEntry($uuid);

        if ($sessionFilters) {
            $params = array_merge($sessionFilters, $request->query->all());
        } else {
            $params = $request->query->all();
        }

        $uri = sprintf('/api/elastic/queries/%s?%s', $id, http_build_query($params));
        return $this->get($uri);
    }

    /**
     * @return array|null
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function getAll() {
        $uri = sprintf('/api/elastic/products?limit=10000');
        return $this->get($uri);
    }
}
