<?php

declare(strict_types=1);

namespace Juweliere\ProductBundle\Controller;

use Juweliere\ProductBundle\ElasticSearch\ElasticParser;
use Juweliere\ProductBundle\ElasticSearch\ElasticProductsParser;
use Psr\Cache\InvalidArgumentException;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;

class WidgetController extends AbstractController
{
    private ElasticProductsParser $productsParser;

    public function __construct( $productsParser)
    {
        $this->productsParser = $productsParser;
    }

    /**
     * @Route("/widgets/{uuid}", name="widgets_show", options={"expose"=true})
     *
     * @param string $uuid
     * @param ElasticParser $widgetParser
     *
     * @return Response
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface|InvalidArgumentException
     */
    public function show(string $uuid, ElasticParser $widgetParser): Response
    {
        $widget = $widgetParser->getSingleWidget($uuid);
        $type = isset($widget['type']['slug']) ? $widget['type']['slug'] : null;
        $template = $this->getWidgetTemplate($type);

        $response = $this->render($template, [
            'widget' => $widget,
            'type' => $type,
            'products' => $this->getProducts($widget),
        ]);

        $response->setPublic();
        $response->setMaxAge(60);

        return $response;
    }

    /**
     * @param string|null $type
     *
     * @return string|null
     */
    private function getWidgetTemplate(?string $type): ?string
    {
        $template = null;

        if ($type) {
            switch ($type) {
                case '3_products':
                    $template = 'widgets/three-products.html.twig';
                    break;

                case '6_products':
                    $template = 'widgets/six-products.html.twig';
                    break;

                case '8_products':
                    $template = 'widgets/eight-products.html.twig';
                    break;

                case 'product_carousel':
                    $template = 'widgets/product-carousel.html.twig';
                    break;

                default:
                    $template = 'widgets/four-products.html.twig';
                    break;
            }
        }

        return $template;
    }

    /**
     * @param array|null $widget
     *
     * @return array|null
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws InvalidArgumentException
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    private function getProducts(?array $widget): ?array
    {
        if (!$widget) {
            return null;
        }

        $products = [];

        foreach ($widget['items'] as $item) {
            $product = $this->productsParser->getSingleProduct($item['product']['sku']);

            if ($product) {
                $products[] = $product;
            }
        }

        return $products;
    }
}
