<?php

namespace Juweliere\ProductBundle\Command;

use Doctrine\ORM\EntityManagerInterface;
use Juweliere\ProductBundle\ElasticSearch\ElasticProductsParser;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use App\Entity\Product;
use App\Session\CatalogFilterManager;
use Symfony\Component\DependencyInjection\ContainerAwareTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;

class JuweliereProductsSyncCommand extends Command
{
    use ContainerAwareTrait;

    private $catalogFilter;
    private $em;
    private ElasticProductsParser $productsParser;

    /**
     * JuweliereProductsSyncCommand constructor.
     * @param CatalogFilterManager $catalogFilter
     * @param EntityManagerInterface $em
     * @param ContainerInterface $container
     * @param ElasticProductsParser $productsParser
     */
    public function __construct(
        CatalogFilterManager $catalogFilter,
        EntityManagerInterface $em,
        ContainerInterface $container,
        ElasticProductsParser $productsParser
    )
    {
        parent::__construct();

        $this->catalogFilter = $catalogFilter;
        $this->em = $em;
        $this->container = $container;
        $this->productsParser = $productsParser;
    }

    protected function configure()
    {
        $this
            ->setName('juweliere:products:sync')
            ->setDescription('Produktimport für Routing')
        ;
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $response = $this->productsParser->getAll();

        if (is_null($response)) {
            return null;
        }

        $sql = 'UPDATE juweliere_product SET is_updated = false';
        $stmt = $this->em->getConnection()->prepare($sql);
        $stmt->execute();

        $counter = 0;
        $counterDisabled = 0;
        foreach ($response['hits'] as $item) {
            if (isset($item['sku'])) {
                try {
                    if(isset($item['reference']) && ($item['isVisible'] == true && $item['isEnabled'] == true)) {
                        $product = $this->isExisting($item['sku']);

                        if (!$product) {
                            $product = new Product();
                        }

                        $product->setSku($item['sku']);
                        $product->setName($item['name']);
                        $product->setReference($item['reference']);
                        $product->setBrand($item['brand']);
                        $product->setLocale('de');
                        $product->setIsEnabled(true);
                        $product->setIsUpdated(true);
                        //                    $product->setProductGroup('uhren');

                        if (!$this->em->isOpen()) {
                            $this->em = $this->em->create(
                                $this->em->getConnection(),
                                $this->em->getConfiguration()
                            );
                        }

                        $this->em->persist($product);
                        $this->em->flush();

                        ++$counter;
                    } else {
                        ++$counterDisabled;
                    }
                } catch (\Exception $e) {
                    $output->writeln("Error: " . $e->getMessage());
                }
            }
        }

        $sql = 'UPDATE juweliere_product SET is_enabled = false WHERE is_updated = false AND is_enabled = true';
        $stmt = $this->em->getConnection()->prepare($sql);
        $stmt->execute();

        $output->writeln('Updated ' . ++$counter . " products");
        $output->writeln('Disabled ' . ++$counter . " products");

        return true;
    }

    private function isExisting(?string $sku): ?Product
    {
        if (!$sku) {
            return null;
        }

        return $this->em->getRepository(Product::class)->findOneBy([
            'sku' => $sku,
        ]);
    }
}
