<?php

declare(strict_types=1);

namespace Juweliere\ProductBundle\Controller;

use Juweliere\ProductBundle\ElasticSearch\ElasticProductsParser;
use Juweliere\ProductBundle\Service\ProductDataHelper;
use Juweliere\ProductBundle\Service\TemplateProvider;
use Juweliere\ProductBundle\Session\SessionManager;
use Sulu\Bundle\ContactBundle\Entity\Contact;
use Sulu\Bundle\HttpCacheBundle\Cache\SuluHttpCache;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Cache\Adapter\AdapterInterface;
use Symfony\Component\HttpClient\HttpClient;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Component\Routing\Annotation\Route;

/**
 * @Route("%juweliere_product_base_url%", name="juweliere_products_products_")
 */
class ProductController extends AbstractController
{
    private $redirectNotAvailable;
    private $productDataHelper;
    private $decoratorBaseUrl;

    public function __construct(
        ProductDataHelper $productDataHelper,
        string            $redirectNotAvailable = "",
        string            $decoratorBaseUrl = ""
    )
    {
        $this->decoratorBaseUrl = $decoratorBaseUrl;
        $this->productDataHelper = $productDataHelper;
        $this->redirectNotAvailable = $redirectNotAvailable;
    }

    /**
     * @Route("/{sku}/related", name="related_products", options={"expose"=true}, requirements={"sku"=".+"}, options={"expose"=true})
     *
     * @param ElasticProductsParser $productsParser
     * @param Request $request
     * @param string $sku
     * @param bool $brand
     * @param bool $small
     * @return Response
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function relatedProducts(
        ElasticProductsParser $productsParser,
        Request               $request,
        TemplateProvider      $templateProvider,
        string                $sku,
        bool                  $brand = false,
        bool                  $small = false
    )
    {
        /** @var array $products */
        $products = $productsParser->getRelatedProducts($sku, true, $brand);

        /** @var string $template */
        $template = $templateProvider->getRelatedProducts($request, $small);

        /** @var string $productGroup */
        $productGroup = "unknown";

        if (isset($products[0])) {
            $productGroup = $this->productDataHelper->getProductType($products[0]);
        }

        $response = $this->render(
            $template,
            [
                'products' => $products,
                'product_group' => $productGroup
            ]
        );

        $response->setMaxAge(0);
        $response->setSharedMaxAge(0);

        $response->headers->set('Cache-Control', 'no-cache, must-revalidate');
        $response->headers->set(
            SuluHttpCache::HEADER_REVERSE_PROXY_TTL,
            $response->getAge()
        );

        return $response;
    }

    /**
     * @Route("/{sku}", name="show", requirements={"sku"=".+"}, options={"expose"=true})
     *
     * @param Request $request
     * @param string $sku
     * @param ElasticProductsParser $productsParser
     * @param ProductDataHelper $productDataHelper
     * @param TemplateProvider $templateProvider
     * @param SessionManager $sessionManager
     *
     * @return Response
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function show(
        Request               $request,
        string                $sku,
        ElasticProductsParser $productsParser,
        ProductDataHelper     $productDataHelper,
        TemplateProvider      $templateProvider,
        SessionManager        $sessionManager): Response
    {
        /** @var array $product */
        $product = $productsParser->getSingleProduct($sku);

        if (is_null($product) || !$product["isEnabled"] || !$product["isVisible"]) {
            return $this->renderError();
        }

        $product = $productDataHelper->consolidateGalleryImages($product);
        $product = $productDataHelper->createTwoColumnDescription($product);

        $response = $this->render($templateProvider->getProductPage($request), [
            'product' => $product,
            'attributes' => $productDataHelper->getAttributes($product),
            'price' => $productDataHelper->getPrice($product),
            'contact_person' => $productDataHelper->getContacData($product),
            'lastseen_products' => $sessionManager->getLastSeenProducts($product)
        ]);

        $response->setSharedMaxAge(86400);
        $response->headers->addCacheControlDirective('must-revalidate', true);

        return $response;
    }

    /**
     * @throws TransportExceptionInterface
     * @throws ServerExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws ClientExceptionInterface
     */
    public function brandLogo(string $brand): ?Response
    {
        $client = HttpClient::createForBaseUri($this->decoratorBaseUrl);

        $response = $client->request('GET', sprintf('/api/juweliere-brands?name=%s', $brand));

        if ($response->getStatusCode() !== 200) {
            return null;
        }

        $content = $response->toArray(false);

        $logo = isset($content['hydra:member'][0]['logo']) ? $content['hydra:member'][0]['logo']['filePath'] : null;

        if (!is_null($logo)) {
            $logo = $this->decoratorBaseUrl . '/images/logos/' . $logo;
        }

        $response = $this->render(
            '@JuweliereProduct/brand_logo.html.twig', [
                'logo' => $logo,
                'brand' => $brand
            ]
        );

        $response->setPublic();
        $response->setSharedMaxAge(86400);

        return $response;
    }

    /**
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|Response
     */
    public function renderError()
    {
        if ($this->redirectNotAvailable !== "") {
            return $this->redirect($this->redirectNotAvailable);
        }

        $content = $this->renderView(
            "@JuweliereProduct/error404.html.twig");
        $response = new Response($content, 404);

        return $response;
    }
}
