<?php

declare(strict_types=1);

namespace Juweliere\ProductBundle\Decorator;

use Juweliere\JuwApiClient\Entity\Order\Order;
use Monolog\Formatter\LineFormatter;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Logger;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;

class DecoratorClient implements DecoratorClientInterface
{
    public function __construct(
        protected HttpClientInterface   $decoratorClient,
        protected LoggerInterface       $logger,
        protected ParameterBagInterface $parameterBag,
    )
    {
        $this->initLogger(
            $this->parameterBag->get('juweliere_product.log.path'),
            $this->parameterBag->get('juweliere_product.log.level'),
        );
        $this->decoratorClient->setLogger($this->logger);
    }

    public function get(string $url): ?array
    {
        try {
            return $this->decoratorClient->request('GET', $url)->toArray(true);
        } catch (\Throwable $exception) {
            $this->logger->error($exception->getMessage());
        }

        return null;
    }

    public function post(string $uri, $data): ?array
    {
        try {
            $response = $this->decoratorClient->request('POST', $uri, [
                'body' => json_encode($data)
            ]);

            $this->logger->info(json_encode($response->getInfo("debug")));

            if (200 === $response->getStatusCode()) {
                return $response->toArray();
            }
        } catch (\Throwable $exception) {
            $this->logger->error($exception->getMessage());
        }

        return null;
    }

    public function createBasket($sku, $qty = 1) {
        return $this->post("/api/products/" . $sku . "/baskets", array("qty" => $qty));
    }

    public function removeBasket($sku, $qty = 1) {
        return $this->post("/api/products/" . $sku . "/baskets/remove", array("qty" => $qty));
    }

    public function createCheckout(Order $order) {
        return $this->post("/api/checkouts", $order);
    }

    private function initLogger($logPath, $logLevel)
    {
        /** @var Logger $logger */
        $logger = new Logger("juw_decorator");

        /** @var RotatingFileHandler $fileHandler */
        $fileHandler = new RotatingFileHandler(
            $logPath . "juw_decorator.log",
            10
        );

        $fileHandler->setLevel($logLevel);

        $fileHandler->setFormatter(new LineFormatter(null, null, true));

        $logger->pushHandler($fileHandler);

        $this->logger = $logger;
    }
}
