<?php

declare(strict_types=1);

namespace Juweliere\ProductBundle\Controller\Website;

use Juweliere\ProductBundle\Decorator\DecoratorClientInterface;
use Juweliere\ProductBundle\ElasticSearch\ElasticProductsParser;
use Juweliere\ProductBundle\Service\ProductDataHelper;
use Juweliere\ProductBundle\Service\TemplateProvider;
use Juweliere\ProductBundle\Session\SessionManager;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;


#[Route("%juweliere_product.prefix%", name: "juweliere_product_products_")]
class ProductController extends AbstractController
{
    public function __construct(
        private ProductDataHelper $productDataHelper,
        private DecoratorClientInterface $decoratorClient,
        private ElasticProductsParser $productsParser,
        private TemplateProvider $templateProvider
    )
    {
    }

    /**
     * @param Request $request
     * @param string $sku
     * @param ElasticProductsParser $productsParser
     * @param ProductDataHelper $productDataHelper
     * @param TemplateProvider $templateProvider
     * @param SessionManager $sessionManager
     * @return Response
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    #[Route("/{sku}", name: "show", requirements: ["sku" => ".+"], options: ["expose" => true])]
    public function show(
        Request               $request,
        string                $sku,
        SessionManager        $sessionManager): Response
    {
        /** @var array $product */
        $product = $this->productsParser->getSingleProduct($sku);

        if (is_null($product) || !$product["isEnabled"] || !$product["isVisible"]) {
            return $this->renderError();
        }

        $product = $this->productDataHelper->consolidateGalleryImages($product);
        $product = $this->productDataHelper->createTwoColumnDescription($product);

        $response = $this->render($this->templateProvider->getProductPage($request), [
            'product' => $product,
            'attributes' => $this->productDataHelper->getAttributes($product),
            'price' => $this->productDataHelper->getPrice($product),
            'lastseen_products' => $sessionManager->getLastSeenProducts($product)
        ]);

        $response->setSharedMaxAge(86400);
        $response->headers->addCacheControlDirective('must-revalidate', true);

        return $response;
    }

    /**
     * @param ElasticProductsParser $productsParser
     * @param Request $request
     * @param TemplateProvider $templateProvider
     * @param string $sku
     * @param bool $brand
     * @param bool $small
     * @return Response
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    #[Route("/{sku}/related", name: "related_products", requirements: ["sku" => ".+"], options: ["expose" => true])]
    public function relatedProducts(
        Request               $request,
        string                $sku,
        bool                  $brand = false,
        bool                  $small = false
    ): Response
    {
        /** @var array $products */
        $products = $this->productsParser->getRelatedProducts($sku, true, $brand);

        $template = $this->templateProvider->getRelatedProducts($request, $small);
        $productGroup = "unknown";

        if (isset($products[0])) {
            $productGroup = $this->productDataHelper->getProductType($products[0]);
        }

        $response = $this->render(
            $template,
            [
                'products' => $products,
                'product_group' => $productGroup
            ]
        );

        $response->setMaxAge(0);
        $response->setSharedMaxAge(0);
        $response->headers->set('Cache-Control', 'no-cache, must-revalidate');

        return $response;
    }

    /**
     * @param string $brand
     * @return Response|null
     */
    public function brandLogo(string $brand): ?Response
    {
        $content = $this->decoratorClient->get(sprintf('/api/juweliere-brands?name=%s', $brand));

        $logo = isset($content['hydra:member'][0]['logo']) ? $content['hydra:member'][0]['logo']['filePath'] : null;

        if (!is_null($logo)) {
            $logo = sprintf("%s/images/logos/%s", $this->getParameter('juweliere_product.base_url.decorator'), $logo);
        }

        $response = $this->render(
            '@JuweliereProduct/pages/product/_logo.html.twig', [
                'logo' => $logo,
                'brand' => $brand
            ]
        );

        $response->setMaxAge(0);
        $response->setSharedMaxAge(0);
        $response->headers->set('Cache-Control', 'no-cache, must-revalidate');

        return $response;
    }

    #[Route("/{sku}/addtocart", name: "addtocart", requirements: ["sku" => ".+"], options: ["expose" => true])]
    public function addToCart(string $sku): ?Response
    {
        /** @var array $product */
        $product = $this->productsParser->getSingleProduct($sku);

        if (is_null($product) || !$product["isEnabled"] || !$product["isVisible"]) {
            return $this->renderError();
        }

        $response = $this->render($this->templateProvider->getAddToCart(), [
            'product' => $product
        ]);

        $response->setMaxAge(0);
        $response->setSharedMaxAge(0);
        $response->headers->set('Cache-Control', 'no-cache, must-revalidate');

        return $response;
    }

    /**
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function renderError(): Response
    {
        $redirectNotAvailable = $this->getParameter('juweliere_product.redirect.notavailable');

        if ($redirectNotAvailable !== "") {
            return $this->redirect($redirectNotAvailable);
        }

        $content = $this->renderView(
            "@JuweliereProduct/error/error404.html.twig");

        return new Response($content, 404);
    }
}
