<?php

declare(strict_types=1);

namespace Juweliere\ProductBundle\Service;

use Carbon\Carbon;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Juweliere\ProductBundle\Service\ProductPriceHelper;

class ProductDataHelper
{
    const PRODUCT_TYPE_WATCH = "uhren";
    const PRODUCT_TYPE_JEWELLERY = "schmuck";
    const PRODUCT_TYPE_ACCESSOIRES = "accessoires";
    const PRODUCT_TYPE_MISC = "sonstiges";

    public function __construct(
        private ParameterBagInterface $parameterBag
    )
    {
    }

    /**
     * @param array $product
     * @return array
     */
    public function consolidateGalleryImages(array $product): array
    {
        if (!empty($product["gallery"])) {
            $images = array();

            if (isset($product["image"])) {
                $images[] = array(
                    "path" => $product["image"]
                );
            }

            usort($product["gallery"], function ($item1, $item2) {
                return $item1['position'] <=> $item2['position'];
            });

            foreach ($product["gallery"] as $image) {
                /** @var string $imagePath */
                $imagePath = explode('_', $product["image"], 2);

                /** @var string $galleryPath */
                $galleryPath = explode('_', $image["path"], 2);

                if (!isset($imagePath[1]) || !isset($galleryPath[1]) || $imagePath[1] != $galleryPath[1]) {
                    $images[] = $image;
                }

            }

            $product["gallery"] = $images;
        }

        return $product;
    }

    public function createTwoColumnDescription(array $product): array
    {
        if ($this->parameterBag->get('juweliere_product.template.description_twocolumns') && isset($product['description']) && !empty($product['description'])) {

            if (preg_match("/^<div/", $product["description"])) {
                $count = substr_count($product['description'], '<div>');

                $product['description'] = str_replace('<span style="font-weight: bold;">', '<strong>', $product['description']);
                $product['description'] = str_replace('</span>', '</strong><br/>', $product['description']);

                $product['description'] = str_replace('<div><span', '<span', $product['description']);
                $product['description'] = str_replace('<br/></div>', '<br/>', $product['description']);
                $product['description'] = str_replace('<br>', '', $product['description']);
                $product['description'] = str_replace('<div>', '', $product['description']);
                $product['description'] = str_replace('</strong><br/></div>', '</strong><br/>', $product['description']);
                $product['description'] = str_replace('</div>', '<br/>', $product['description']);
                $product['description'] = str_replace('<br/><br/>', '<br/>', $product['description']);

                $count = substr_count($product['description'], '<strong>');

                $maxItemsPerCol = (int)round(($count) / 2);

                $count = 0;

                $description = '<div class="col-md-6">';
                foreach (explode("<strong>", $product['description']) as $line) {
                    if (!empty(trim($line))) {
                        $description .= '<p><strong>' . $line . "</p>";
                        $count++;
                        if ($count == $maxItemsPerCol) {
                            $description .= '</div><div class="col-md-6">';
                            #$count = 0;
                        }
                    }

                }

                $description .= '</div>';
            } else {
                $product['description'] = str_replace('<p style="text-align: left;">', '', $product['description']);
                $product['description'] = str_replace('<p class="textbody">', '', $product['description']);

                if (preg_match("/span class=\"strongemphasis\"/", $product['description'])) {
                    $product['description'] = str_replace('<span class="strongemphasis">', '', $product['description']);
                    $product['description'] = str_replace('</span></p><p>', '<br />', $product['description']);
                }

                $product['description'] = str_replace('</p>', '', $product['description']);
                $product['description'] = str_replace('<p>', '', $product['description']);
                $product['description'] = str_replace('<strong><br />', '<strong>', $product['description']);

                // /products/tudor-m12313-0003
                if (preg_match("/span style=\"font-weight: bold;\"/", $product['description'])) {
                    $product['description'] = str_replace('span style="font-weight: bold;"', 'strong', $product['description']);
                    $product['description'] = str_replace('<span style="font-weight: 700;"><br></span>', '<br>', $product['description']);

                    $product['description'] = str_replace('/span', '/strong', $product['description']);
                }

                $count = substr_count($product['description'], '<strong>');

                $maxItemsPerCol = (int)round(($count) / 2);

                $count = 0;

                $description = '<div class="col-md-6">';
                foreach (explode("<strong>", $product['description']) as $line) {
                    if (!empty(trim($line))) {
                        $description .= '<p><strong>' . nl2br($line) . "</p>";
                        $count++;
                        if ($count == $maxItemsPerCol) {
                            $description .= '</div><div class="col-md-6">';
                            #$count = 0;
                        }
                    }

                }

                $description .= '</div>';
            }

            $product["description"] = $description;
        }

        return $product;
    }

    public function getAttributes($product): array
    {
        $attributes = [];

        if (isset($product['attributes']) && sizeof($product['attributes'])) {
            foreach ($product['attributes'] as $filter) {
                if ($filter['isFilter']) {
                    $attributes[] = $filter;
                }
            }
        }

        return $attributes;
    }

    public function getProductBrand($product): ?string
    {
        if (!empty($product['attributes'])) {
            foreach ($product["attributes"] as $attribute) {
                if ($attribute["attributeCode"] == "juweliere_brand") {
                    return $attribute["valueCode"];
                }
            }
        }

        return null;
    }

    public function getProductType($product): string
    {
        if (!empty($product['attributes'])) {
            foreach ($product["attributes"] as $attribute) {
                if ($attribute["attributeCode"] == "schmuckart") {
                    return self::PRODUCT_TYPE_JEWELLERY;
                }
            }
        }

        return self::PRODUCT_TYPE_WATCH;
    }

    /**
     * @param array|null $product
     * @param bool $currencyAsText
     * @return array|null
     */
    public function getPrice(?array $product, bool $currencyAsText = false): ?array
    {
        if (!$product) {
            return null;
        }

        $showPrice = (bool)$product['showPrice'];
        if (!$showPrice) {
            return null;
        }

        $rrp = isset($product['rrp']) ? (float)$product['rrp'] : null;
        $currencySymbol = $currencyAsText ? 'EUR' : '€';

        $data = [
            'rrp' => [
                'value' => $rrp,
                'currency' => $currencySymbol,
            ],
            'specialPrice' => null,
            'actualPrice' => [
                'value' => $rrp,
                'currency' => '€',
            ],
        ];

        $specialsPrices = isset($product['prices']) && sizeof($product['prices']) > 0 ? $product['prices'] : null;

        if ($specialsPrices) {
            foreach ($specialsPrices as $specialsPrice) {
                $validFrom = new Carbon($specialsPrice['validFrom']);
                $validThrough = new Carbon($specialsPrice['validThrough']);

                $now = Carbon::now();

                if ($validFrom->lte($now) && $validThrough->gte($now)) {
                    if ($specialsPrice['currency'] === $data['actualPrice']['currency']) {
                        $data['specialPrice'] = [
                            'value' => (float)$specialsPrice['price'],
                            'currency' => $specialsPrice['currency'],
                        ];

                        $data['actualPrice']['value'] = (float)$specialsPrice['price'];
                    }
                }
            }
        }

        return $data;
    }
}
