<?php

declare(strict_types=1);

namespace Juweliere\ProductBundle\Controller\Website;

use Juweliere\ProductBundle\ElasticSearch\ElasticProductsParser;
use Juweliere\ProductBundle\Session\CatalogFilterManager;
use Psr\Cache\InvalidArgumentException;
use Sulu\Component\Content\Compat\StructureInterface;
use Sulu\Component\DocumentManager\DocumentManagerInterface;
use Sulu\Component\DocumentManager\Exception\DocumentManagerException;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;
use Symfony\Component\Cache\Adapter\AdapterInterface;

class SelectionController extends WebsiteController
{
    private ElasticProductsParser $productsParser;
    private CatalogFilterManager $filterManager;
    private DocumentManagerInterface $documentManager;
    private AdapterInterface $cache;
    private HttpClientInterface $decoratorClient;

    public function __construct(
        HttpClientInterface $decoratorClient,
        DocumentManagerInterface $documentManager,
        ElasticProductsParser $productsParser,
        CatalogFilterManager $filterManager,
        AdapterInterface $cache
    )
    {
        $this->productsParser = $productsParser;
        $this->filterManager = $filterManager;
        $this->documentManager = $documentManager;
        $this->cache = $cache;
        $this->decoratorClient = $decoratorClient;
    }

    /**
     * @param Request $request
     * @param StructureInterface $structure
     * @param bool $preview
     * @param bool $partial
     *
     * @return Response
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws InvalidArgumentException
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     * @throws \Throwable
     */
    public function index(Request $request, StructureInterface $structure, bool $preview = false, bool $partial = false): Response
    {
        $this->updateFilters($request, $structure);

        $queryId = $structure->getPropertyValue('selection_uuid');

        if (is_null($queryId)) {
            return $this->renderStructure(
                $request,
                $structure,
                [],
                $preview,
                $partial
            );
        }

        $content = $this->productsParser->getProductQuery($request, $queryId);

        $attributes = [];

        if ($aggs = $content['aggregations'] ?? null) {
            foreach ($aggs as $agg) {
                $cachedAttribute = $this->cache->getItem('filter_'.$agg['key']);
                $attribute = null;

                if ($cachedAttribute->isHit()) {
                    $attribute = $cachedAttribute->get();
                } else {
                    $attributeResponse = $this->decoratorClient->request('GET', sprintf('/api/elastic/attributes/%s', $agg['key']));

                    if ($attributeResponse->getStatusCode() === 200) {
                        $attribute = $attributeResponse->toArray(false);
                        $cachedAttribute->set($attribute);
                        $this->cache->save($cachedAttribute);
                    }
                }

                $attributes[] = [
                    'attribute' => $attribute,
                    'options' => $this->bucketLabels($attribute, $agg['value']['buckets']),
                ];
            }
        }

        if ($queryId) {
            return $this->renderStructure(
                $request,
                $structure,
                [
                    'selection_uuid' => $structure->getPropertyValue('selection_uuid'),
                    'result' => $this->productsParser->getProductQuery($request, $queryId),
                    'query' => $queryId,
                    'uuid' => $structure->getUuid(),
                    'attributes' => $attributes,
                ],
                $preview,
                $partial
            );
        }

        return $this->renderStructure(
            $request,
            $structure,
            [],
            $preview,
            $partial
        );
    }

    /**
     * @Route("/api/selections/{uuid}/grid", name="juweliere_catalog_selection_grid", options={"expose"=true})
     *
     * @param Request $request
     * @param string $uuid
     *
     * @return Response
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function selectionsGrid(Request $request, string $uuid): Response
    {
        $page = $request->query->get('page');

        if ($page) {
            $this->filterManager->setPage($uuid, (int) $page);
        }

        return $this->render('pages/catalog/_grid.html.twig', [
            'result' => $this->productsParser->getProductQuery($request, $uuid),
            'uuid' => $uuid,
        ]);
    }

    /**
     * @Route("/api/session/catalog/{uuid}", name="juweliere_catalog_session_add_product_filter")
     *
     * @param Request $request
     * @param string $uuid
     *
     * @return JsonResponse
     */
    public function addFilterToSession(Request $request, string $uuid): JsonResponse
    {
        $filters = $request->query->get('filters');

        if ($filters) {
            foreach ($filters as $key => $filter) {
                $data = [
                    'code' => $key,
                    'value' => $filter,
                ];
                $this->filterManager->addFilter($uuid, $data);
            }

            $this->filterManager->setPage($uuid, 1);
        }

        return new JsonResponse($this->filterManager->getEntry($uuid), 204);
    }

    /**
     * @Route("/api/session/catalog/{uuid}/delete/all", name="juweliere_catalog_session_delete_all_product_filters")
     *
     * @param string $uuid
     *
     * @return JsonResponse
     */
    public function deleteFiltersFromSession(string $uuid): JsonResponse
    {
        $this->filterManager->removeEntry($uuid);

        return new JsonResponse(null, 204);
    }

    /**
     * @Route("/api/session/catalog/{uuid}/delete/{filter}", name="juweliere_catalog_session_delete_product_filter")
     *
     * @param string $uuid
     * @param string $filter
     *
     * @return RedirectResponse
     * @throws DocumentManagerException
     */
    public function deleteFilterFromSession(string $uuid, string $filter): RedirectResponse
    {
        $this->filterManager->removeFilter($uuid, $filter);

        $document = $this->documentManager->find($uuid);

        return new RedirectResponse($document->getResourceSegment());
    }

    /**
     * @param Request $request
     * @param StructureInterface $structure
     */
    private function updateFilters(Request $request, StructureInterface $structure): void
    {
        $uuid = $structure->getUuid();
        $filters = $request->query->get('filters');
        $page = $request->query->get('page');
        $limit = $request->query->get('limit');
        $sort = $request->query->get('sort');
        $dir = $request->query->get('dir', 'asc');

        if ($filters) {
            foreach ($filters as $key => $filter) {
                $data = [
                    'code' => $key,
                    'value' => $filter,
                ];
                $this->filterManager->addFilter($uuid, $data);
            }
        }

        if ($page) {
            $this->filterManager->setPage($uuid, (int) $page);
        }

        if ($limit) {
            $this->filterManager->setLimit($uuid, (int) $limit);
        }

        if ($sort) {
            $this->filterManager->setSorting($uuid, $sort, $dir);
        }
    }

    /**
     * @throws InvalidArgumentException
     */
    protected function bucketLabels($attribute, array $buckets): array
    {
        $options = [];

        foreach ($buckets as $bucket) {
            $cachedOption = $this->cache->getItem('option_'.$bucket['key']);

            if ($cachedOption->isHit()) {
                $options[] = [
                    'key' => $bucket['key'],
                    'label' => $cachedOption->get(),
                ];
            } else {
                foreach ($attribute['options'] as $option) {
                    if ($option['code'] === $bucket['key']) {
                        $options[] = [
                            'key' => $bucket['key'],
                            'label' => $option['label'],
                        ];

                        $cachedOption->set($option['label']);
                        $this->cache->save($cachedOption);
                    }
                }
            }
        }

        return $options;
    }
}
