<?php

declare(strict_types=1);

namespace Juweliere\ProductBundle\Service;

use Carbon\Carbon;
use Juweliere\ProductBundle\Decorator\DecoratorClientInterface;

class ProductPriceHelper
{
    public function __construct(private DecoratorClientInterface $decoratorClient)
    {
    }

    /**
     * @param array|null $product
     * @param bool $currencyAsText
     * @return array|null
     */
    public function getPrice(?array $product, bool $currencyAsText = false): ?array
    {
        if (!$product) {
            return null;
        }

        $showPrice = isset($product['showPrice']) ? (bool)$product['showPrice'] : true;
        if (!$showPrice) {
            return null;
        }

        $rrp = isset($product['rrp']) ? (float)$product['rrp'] : null;
        $currencySymbol = $currencyAsText ? 'EUR' : '€';

        $data = [
            'rrp' => [
                'value' => $rrp,
                'currency' => $currencySymbol,
            ],
            'specialPrice' => null,
            'actualPrice' => [
                'value' => $rrp,
                'currency' => $currencySymbol,
            ],
        ];

        $response = $this->decoratorClient->get(sprintf('/api/juweliere-product-prices?product.sku=%s&t=' . time(), $product['sku']));
        $specialPrices = $response['hydra:member'] ?? null;


        if ($specialPrices) {
            foreach ($specialPrices as $specialPrice) {
                $validFrom = new Carbon($specialPrice['validFrom']);
                $validThrough = new Carbon($specialPrice['validThrough']);

                $now = Carbon::now();

                if ($validFrom->lte($now) && $validThrough->gte($now)) {
                    $data['specialPrice'] = [
                        'value' => (float)$specialPrice['price'],
                        'currency' => $specialPrice['currency'],
                    ];

                    $data['actualPrice']['value'] = (float)$specialPrice['price'];
                }
            }
        }

        return $data;
    }
}
