<?php

declare(strict_types=1);

namespace Juweliere\ProductBundle\Twig;

use Juweliere\ProductBundle\ElasticSearch\ElasticProductsParser;
use Juweliere\ProductBundle\Service\ProductPriceHelper;
use Twig\Environment;
use Twig\Error\LoaderError;
use Twig\Error\RuntimeError;
use Twig\Error\SyntaxError;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

class ProductExtension extends AbstractExtension
{
    const PRODUCT_CDN = 'https://bilder.juweliere.de/cdn';

    public function __construct(private ElasticProductsParser $productsParser, private ProductPriceHelper $priceHelper)
    {
    }

    /**
     * @return array
     */
    public function getFunctions(): array
    {
        return [
            'product_image' => new TwigFunction('product_image', [$this, 'getProductImage'], ['is_safe' => ['html']]),
            'product_teaser' => new TwigFunction('product_teaser', [$this, 'getProductTeaser'], ['needs_environment' => true, 'is_safe' => ['html']]),
            'price_display' => new TwigFunction('price_display', [$this, 'getProductPrice']),
            'product_price' => new TwigFunction('product_price', [$this, 'getProductPrice']),
        ];
    }

    public function getProductImage(string $image, array $parameters = []): string
    {
        $size = $parameters['size'] ?? 300;

        return sprintf('%s/%sx%s/%s', self::PRODUCT_CDN, $size, $size, $image);
    }

    /**
     * @throws SyntaxError
     * @throws RuntimeError
     * @throws LoaderError
     */
    public function getProductTeaser(Environment $environment, $sku, $tracking = "Produkt Teaser"): ?string
    {
        $product = $this->productsParser->getSingleProduct($sku);

        if ($product) {
            return $environment->render('@JuweliereProduct/pages/product/_preview.html.twig', [
                'product' => $product,
                'tracking' => $tracking
            ]);
        }

        return null;
    }

    public function getProductPrice(array $product, bool $displayDiff = true): ?array
    {
        try {
            $prices = $this->priceHelper->getPrice($product);

            if ($prices['actualPrice']['value'] === 0 or !$prices['actualPrice']['value']) {
                return null;
            }

            $diff = null;
            if ($displayDiff) {
                $diff = $prices['rrp']['value'] - $prices['actualPrice']['value'];
            }

            return array(
                "rrp" => $prices['rrp']['value'],
                "display" => $prices['actualPrice']['value'],
                "diff" => $diff,
            );

        } catch (\Throwable $exception) {
            return null;
        }
    }
}
