<?php

declare(strict_types=1);

namespace Juweliere\ProductBundle\Controller\Website;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Juweliere\ProductBundle\ElasticSearch\ElasticProductsParser;
use Massive\Bundle\SearchBundle\Search\SearchManagerInterface;
use Sulu\Bundle\MediaBundle\Media\Manager\MediaManagerInterface;
use Sulu\Component\Webspace\Analyzer\RequestAnalyzerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Cache\Adapter\FilesystemAdapter;
use Symfony\Component\Cache\Adapter\RedisAdapter;
use Symfony\Component\Cache\CacheItem;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\KernelInterface;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;

class SearchController extends AbstractController
{
    const ROLEX_SEARCH_CACHE_KEY = "search.rolex";

    private ElasticProductsParser $productsParser;
    private HttpClientInterface $rolexClient;
    private MediaManagerInterface $mediaManager;
    private ParameterBagInterface $parameterBag;


    public function __construct(
        ElasticProductsParser $productsParser,
        HttpClientInterface $rolexClient,
        MediaManagerInterface $mediaManager,
        ParameterBagInterface $parameterBag
    )
    {
        $this->productsParser = $productsParser;
        $this->rolexClient = $rolexClient;
        $this->mediaManager = $mediaManager;
        $this->parameterBag = $parameterBag;
    }

    /**
     * @Route("%juweliere_product.search_uri%", name="search", options={"expose"=true})
     *
     * @param Request $request
     *
     * @return Response
     *
     * @throws TransportExceptionInterface
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     */
    public function search(
        Request $request,
        ElasticProductsParser $productsParser,
        RequestAnalyzerInterface $requestAnalyzer,
        SearchManagerInterface $searchManager,
        KernelInterface $kernel
    ): Response
    {
        /** @var string|null $query */
        $query = $request->query->get('q');
        $page = $request->query->get('page', 1);
        $limit = $request->query->get('limit', 24);

        if (!$query) {
            return $this->render('pages/search.html.twig', [
            ]);
        }

        /**
         * Special Rolex handling
         */
        if ($this->isRolexSearch($query)) {
            return $this->redirect("/rolex");
        }

        /**
         * Special Wellendorff handling
         */
        if ($this->isWellendorf($query)) {
            return $this->redirect("/wellendorff");
        }

        $locale = $requestAnalyzer->getCurrentLocalization()->getLocale();
        $webspace = $requestAnalyzer->getWebspace();

        $queryString = '';
        if (\strlen($query) < 3) {
            $queryString .= '+("' . self::escapeDoubleQuotes($query) . '") ';
        } else {
            $queryValues = \explode(' ', $query);
            foreach ($queryValues as $queryValue) {
                if (\strlen($queryValue) > 2) {
                    $queryString .= '+("' . self::escapeDoubleQuotes($queryValue) . '" OR ' .
                        \preg_replace('/([^\pL\s\d])/u', '?', $queryValue) . '* OR ' .
                        \preg_replace('/([^\pL\s\d])/u', '', $queryValue) . '~) ';
                } else {
                    $queryString .= '+("' . self::escapeDoubleQuotes($queryValue) . '") ';
                }
            }
        }

        /** @var array $indexes */
        $indexes = explode(",", $this->parameterBag->get('search_indexes'));

        try {
            $pages = $searchManager
                ->createSearch($queryString)
                ->locale($locale)
                ->indexes(
                    \str_replace(
                        '#webspace#',
                        $webspace->getKey(),
                        $indexes
                    )
                )
                ->setLimit(128)
                ->execute();
        } catch (\Throwable $e) {
            if ($kernel->getEnvironment() === "dev") {
                var_dump($e->getMessage());
                die();
            }
        }

        /*echo "<pre>";
        foreach ($pages as $hit) {
            var_dump($hit->getDocument()->getTitle());
            var_dump($hit->getDocument()->getDescription());
            var_dump($hit->getDocument()->getUrl());
            var_dump($hit->getDocument()->getImageUrl());
            var_dump($hit->getDocument()->getIndex());
            #var_dump($hit->getDocument()->getFields());
            #var_dump($hit->getScore());
            #var_dump($hit);
        }
        echo "</pre>";
        die();*/

        foreach ($pages as $hit) {
            try {

                $excerptImag = json_decode($hit->getDocument()->getFields()["excerptIcon"]->getValue());

                if ($hit->getDocument()->getFields()["_structure_type"]->getValue() == "brand" &&
                    !empty($hit->getDocument()->getImageUrl())
                ) {
                    $hit->getDocument()->setImageUrl(str_replace("sulu-100x100", "100x", $hit->getDocument()->getImageUrl()));
                } elseif (!empty($excerptImag->ids)) {
                    $hit->getDocument()->setImageUrl($excerptImag->ids[0]);
                }

                if (!empty(trim($hit->getDocument()->getFields()["excerptTitle"]->getValue()))) {
                    $hit->getDocument()->setTitle(trim($hit->getDocument()->getFields()["excerptTitle"]->getValue()));
                }
                if (!empty(trim($hit->getDocument()->getFields()["excerptDescription"]->getValue()))) {
                    $hit->getDocument()->setDescription(trim($hit->getDocument()->getFields()["excerptDescription"]->getValue()));
                }

                if (!empty($hit->getDocument()->getImageUrl())) {
                    $imgUrl = $this->mediaManager->getById($hit->getDocument()->getImageUrl(), "de")->getFormats()["sulu-100x100"];
                    $hit->getDocument()->setImageUrl($imgUrl);
                }
            } catch (\Throwable $e) {
                if ($kernel->getEnvironment() === "dev") {
                    var_dump($e->getMessage());
                    die();
                }
            }
        }

        /** @var array $products */
        $products = $productsParser->search($query, $page, $limit);

        $response = $this->render('pages/search.html.twig', [
            'products' => $products,
            'pages' => $pages,
            'query' => $query
        ]);

        $response->setSharedMaxAge(86400);
        $response->headers->addCacheControlDirective('must-revalidate', true);

        return $response;
    }

    /**
     * @param $query
     * @return bool
     * @throws \GuzzleHttp\Exception\GuzzleException
     * @throws \Psr\Cache\InvalidArgumentException
     */
    private function isRolexSearch($query) {
        /** @var RedisAdapter $cache */
        $cache = new FilesystemAdapter();

        /** @var CacheItem $rolexSearch */
        $rolexSearch = $cache->getItem(self::ROLEX_SEARCH_CACHE_KEY);

        if (!$rolexSearch->isHit())
        {
            $rolexSearch->expiresAfter(3600);

            /** @var Client $client */
            $client = new Client();

            try {
                $result = $this->rolexClient->request('GET', '/api/keywords.json');
            } catch (RequestException $e) {
                return false;
            }

            /** @var object $rolexSearches */
            $rolexSearches = json_decode($result->getContent());

            foreach ($rolexSearches as $item) {
                $keywords[] = $item->keyword;
            }

            /** @var string $rolexSearchString */
            $rolexSearchString = implode("|", $keywords);

            $rolexSearch->set($rolexSearchString);
            $cache->save($rolexSearch);
        }

        if (preg_match("/(" . $rolexSearch->get() . ")/i", $query)) {
            return true;
        };

        return false;
    }

    /**
     * @param $query
     * @return bool
     */
    private function isWellendorf($query)
    {
        if (preg_match("/wellendorff/", strtolower($query))) {
            return true;
        }
        return false;
    }

    /**
     * Returns the string with escaped quotes.
     *
     * @param string $query
     *
     * @return string
     */
    private static function escapeDoubleQuotes($query)
    {
        return \str_replace('"', '\\"', $query);
    }
}
