<?php

declare(strict_types=1);

namespace Juweliere\ProductBundle\Controller\Website;

use Juweliere\ProductBundle\Decorator\DecoratorClientInterface;
use Juweliere\ProductBundle\ElasticSearch\ElasticProductsParser;
use Juweliere\ProductBundle\Service\NotFoundHelper;
use Juweliere\ProductBundle\Service\ProductDataHelper;
use Juweliere\ProductBundle\Service\ProductPriceHelper;
use Juweliere\ProductBundle\Service\TemplateProvider;
use Juweliere\ProductBundle\Session\SessionManager;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;


#[Route("%juweliere_product.prefix%", name: "juweliere_product_products_")]
class ProductController extends AbstractController
{
    public function __construct(
        private ProductDataHelper $productDataHelper,
        private DecoratorClientInterface $decoratorClient,
        private ElasticProductsParser $productsParser,
        private TemplateProvider $templateProvider
    )
    {
    }

    /**
     * @param Request $request
     * @param string  $sku
     * @param bool    $brand
     * @param bool    $small
     * @param string  $headline
     *
     * @return Response
     */
    #[Route("/{sku}/related", name: "related_products", requirements: ["sku" => ".+"], options: ["expose" => true])]
    public function relatedProducts(
        Request               $request,
        string                $sku,
        bool                  $brand = false,
        bool                  $small = false,
        string                $headline = ""
    ): Response
    {
        $random =(bool)$request->get("random", true);

        /** @var array $products */
        $products = $this->productsParser->getRelatedProducts($sku, $random, $brand);

        $template = $this->templateProvider->getRelatedProducts($request, $small);
        $productGroup = "unknown";

        if (isset($products[0])) {
            $productGroup = $this->productDataHelper->getProductType($products[0]);
        }

        $response = $this->render(
            $template,
            [
                'products' => $products,
                'headline' => $headline,
                'product_group' => $productGroup
            ]
        );

        $response->setMaxAge(0);
        $response->setSharedMaxAge(0);
        $response->headers->set('Cache-Control', 'no-cache, must-revalidate');

        return $response;
    }

    /**
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    #[Route("/{sku}/addtocart", name: "addtocart", requirements: ["sku" => ".+"], options: ["expose" => true])]
    public function addToCart(string $sku): ?Response
    {
        /** @var array $product */
        $product = $this->productsParser->getSingleProduct($sku);

        if (is_null($product) || !$product["isEnabled"] || !$product["isVisible"]) {
            return $this->renderError();
        }

        $response = $this->render($this->templateProvider->getAddToCart(), [
            'product' => $product
        ]);

        $response->setMaxAge(0);
        $response->setSharedMaxAge(0);
        $response->headers->set('Cache-Control', 'no-cache, must-revalidate');

        return $response;
    }

    /**
     * @param Request        $request
     * @param string         $sku
     * @param SessionManager $sessionManager
     * @param NotFoundHelper $notFoundHelper
     *
     * @return Response
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    #[Route("/{sku}", name: "show", requirements: ["sku" => ".+"], options: ["expose" => true])]
    public function show(
        Request               $request,
        string                $sku,
        SessionManager        $sessionManager,
        NotFoundHelper        $notFoundHelper
    ): Response
    {
        /** @var array $product */
        $product = $this->productsParser->getSingleProduct($sku);

        if (!is_null($product) && isset($product["slug"]) &&  $product["slug"] !== $sku) {
            return $this->redirect($this->getParameter("juweliere_product.prefix") . "/" . $product["slug"], 301);
        }

        if (is_null($product) || !$product["isEnabled"] || !$product["isVisible"]) {

            $alternatePage = $notFoundHelper->getAlternateBrandPage($sku);
            return $this->renderError($alternatePage);
        }

        $product = $this->productDataHelper->consolidateGalleryImages($product);
        $product = $this->productDataHelper->createTwoColumnDescription($product);
        $attributes = $this->productDataHelper->getAttributes($product);
        $allAttributes = $this->productDataHelper->getAllAttributes($product);

        $customPDP = $this->templateProvider->getCustomPDP($product, $allAttributes);

        $response = $this->render($this->templateProvider->getProductPage($customPDP), [
            'customPDP' => $customPDP,
            'product' => $product,
            'attributes' => $attributes,
            'all_attributes' => $allAttributes,
            'price' => $this->productDataHelper->getPrice($product),
            'lastseen_products' => $sessionManager->getLastSeenProducts($product)
        ]);

        $response->setSharedMaxAge(86400);
        $response->headers->addCacheControlDirective('must-revalidate', true);

        return $response;
    }

    /**
     * @param string $brand
     * @return Response|null
     */
    public function brandLogo(string $brand): ?Response
    {
        $content = $this->decoratorClient->get(sprintf('/api/juweliere-brands?name=%s&isEnabled=true', $brand));

        $logo = isset($content['hydra:member'][0]['logo']) ? $content['hydra:member'][0]['logo']['filePath'] : null;

        if (!is_null($logo)) {
            $logo = sprintf("%s/images/logos/%s", $this->getParameter('juweliere_product.base_url.decorator_public'), $logo);
        }

        $response = $this->render(
            '@JuweliereProduct/pages/product/_logo.html.twig', [
                'logo' => $logo,
                'brand' => $brand
            ]
        );

        $response->setMaxAge(0);
        $response->setSharedMaxAge(0);
        $response->headers->set('Cache-Control', 'no-cache, must-revalidate');

        return $response;
    }

    /**
     * @param string $sku
     * @return Response|null
     */
    public function availableQty(string $sku): ?Response
    {
        $content = $this->decoratorClient->get(sprintf('/api/products/%s/qty', $sku));

        $response = $this->render(
            '@JuweliereProduct/pages/product/_stock_status.html.twig', [
                'qty' => $content['qty'],
            ]
        );

        $response->setMaxAge(0);
        $response->setSharedMaxAge(0);
        $response->headers->set('Cache-Control', 'no-cache, must-revalidate');

        return $response;
    }

    /**
     * @param array $product
     * @return Response|null
     */
    public function checkoutButtons(array $product): ?Response
    {
        $product["qty"] = $this->decoratorClient->get(sprintf('/api/products/%s/qty', $product["sku"]))["qty"];

        $response = $this->render(
            '@JuweliereProduct/pages/product/_checkout_buttons.html.twig', [
                'product' => $product,
            ]
        );

        $response->setMaxAge(0);
        $response->setSharedMaxAge(0);
        $response->headers->set('Cache-Control', 'no-cache, must-revalidate');

        return $response;
    }

    /**
     * @param ProductPriceHelper $productPriceHelper
     * @param Request $request
     * @param string $sku
     * @return Response|null
     */
    public function productPrice(ProductPriceHelper $productPriceHelper, Request $request, string $sku): ?Response
    {
        $displayDiff = $request->get('displayDiff', false);
        $product = $this->decoratorClient->get(sprintf('/api/juweliere-products/%s', $sku));

        if (!$product) {
            $response = new Response();
            $response->setContent('');
        }

        $prices = $productPriceHelper->getPrice($product, true);

        if (is_null($prices)) {
            $response = new Response();
            $response->setContent('');
        } else {

            if ($prices['actualPrice']['value'] === 0 or !$prices['actualPrice']['value']) {
                return null;
            }

            $diff = null;
            if ($displayDiff) {
                $diff = $prices['rrp']['value'] - $prices['actualPrice']['value'];
            }

            $response = $this->render(
                '@JuweliereProduct/pages/product/_price.html.twig', [
                    "rrp" => $prices['rrp']['value'],
                    "current" => $prices['actualPrice']['value'],
                    "diff" => $diff
                ]
            );
        }

        $response->setMaxAge(0);
        $response->setSharedMaxAge(0);
        $response->headers->set('Cache-Control', 'no-cache, must-revalidate');

        return $response;
    }

    /**
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function renderError($alternatePage = null): Response
    {
        $redirectEnabled = (bool) $this->getParameter('juweliere_product.redirect.enabled');
        $redirectNotAvailable = $this->getParameter('juweliere_product.redirect.notavailable');

        // Wenn eine alternative Seite als Ziel angegeben wird,
        // wird in jedem Fall dorthin umgeleitet.
        if (true === $redirectEnabled && $alternatePage) {
            return $this->redirect($alternatePage . "?alert=info&msg=product_redirect", 301);
        }

        // Wenn in der Bundle-Konfiguration eine globale Umleitung aktiviert und ein Ziel hinterlegt ist,
        // wird dorthin umgeleitet
        //
        // Achtung: Dies sollte aus SEO-Gründen die Ausnahme sein!
        if ($redirectNotAvailable !== "" && true === $redirectEnabled) {
            return $this->redirect($redirectNotAvailable, 301);
        }

        // Darstellung eines 404-Templates
        $content = $this->renderView("@JuweliereProduct/error/error404.html.twig");

        return new Response($content, 404);
    }
}
