<?php

declare(strict_types=1);

namespace Juweliere\ProductBundle\Controller\Website;

use Juweliere\ProductBundle\ElasticSearch\ElasticProductsParser;
use Juweliere\ProductBundle\Session\CatalogFilterManager;
use Psr\Cache\InvalidArgumentException;
use Sulu\Component\Content\Compat\StructureInterface;
use Sulu\Component\DocumentManager\DocumentManagerInterface;
use Sulu\Component\DocumentManager\Exception\DocumentManagerException;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;
use Psr\Cache\CacheItemPoolInterface;

class SelectionController extends WebsiteController
{
    public function __construct(
        private readonly HttpClientInterface      $decoratorClient,
        private readonly DocumentManagerInterface $documentManager,
        private readonly ElasticProductsParser    $productsParser,
        private readonly CatalogFilterManager     $filterManager,
        private readonly CacheItemPoolInterface $cache
    )
    {
    }

    /**
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws InvalidArgumentException
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     * @throws \Throwable
     */
    public function index(Request $request, StructureInterface $structure, bool $preview = false, bool $partial = false): Response
    {
        $queryId = $structure->getPropertyValue('selection_uuid');

        try {
            $defaultLimit = $structure->getPropertyValue('selection_limit_per_page');
        } catch (\Throwable) {
            $defaultLimit = 12;
        }

        if ($defaultLimit < 1) {
            $defaultLimit = 12;
        }

        $this->updateFilters($request, $structure, $defaultLimit);

        if (is_null($queryId)) {
            return $this->renderStructure(
                $request,
                $structure,
                [],
                $preview,
                $partial
            );
        }

        $content = $this->productsParser->getProductQuery($request, $queryId);

        $attributes = [];

        if ($aggs = $content['aggregations'] ?? null) {
            foreach ($aggs as $agg) {
                $cachedAttribute = $this->cache->getItem('filter_'.$agg['key']);
                $attribute = null;

                if ($cachedAttribute->isHit()) {
                    $attribute = $cachedAttribute->get();
                } else {
                    $attributeResponse = $this->decoratorClient->request('GET', sprintf('/api/elastic/attributes/%s', $agg['key']));

                    if ($attributeResponse->getStatusCode() === 200) {
                        $attribute = $attributeResponse->toArray(false);
                        $cachedAttribute->set($attribute);
                        $this->cache->save($cachedAttribute);
                    }
                }

                $attributes[] = [
                    'attribute' => $attribute,
                    'options' => $this->bucketLabels($attribute, $agg['value']['buckets']),
                ];
            }
        }

        if ($queryId) {
            return $this->renderStructure(
                $request,
                $structure,
                [
                    'selection_uuid' => $structure->getPropertyValue('selection_uuid'),
                    'result' => $this->productsParser->getProductQuery($request, $queryId),
                    'query' => $queryId,
                    'uuid' => $structure->getUuid(),
                    'attributes' => $attributes,
                ],
                $preview,
                $partial
            );
        }

        return $this->renderStructure(
            $request,
            $structure,
            [],
            $preview,
            $partial
        );
    }

    /**
     *
     *
     *
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    #[Route(path: '/api/selections/{uuid}/grid', name: 'juweliere_catalog_selection_grid', options: ['expose' => true])]
    public function selectionsGrid(Request $request, string $uuid): Response
    {
        $page = $request->query->get('page');

        if ($page) {
            $this->filterManager->setPage($uuid, (int) $page);
        }

        return $this->render('pages/catalog/_grid.html.twig', [
            'result' => $this->productsParser->getProductQuery($request, $uuid),
            'uuid' => $uuid,
        ]);
    }


    #[Route(path: '/api/session/catalog/{uuid}', name: 'juweliere_catalog_session_add_product_filter')]
    public function addFilterToSession(Request $request, string $uuid): JsonResponse
    {
        $filters = $request->query->get('filters');

        if ($filters) {
            foreach ($filters as $key => $filter) {
                $data = [
                    'code' => $key,
                    'value' => $filter,
                ];
                $this->filterManager->addFilter($uuid, $data);
            }

            $this->filterManager->setPage($uuid, 1);
        }

        return new JsonResponse($this->filterManager->getEntry($uuid), Response::HTTP_NO_CONTENT);
    }


    #[Route(path: '/api/session/catalog/{uuid}/delete/all', name: 'juweliere_catalog_session_delete_all_product_filters')]
    public function deleteFiltersFromSession(string $uuid): JsonResponse
    {
        $this->filterManager->removeEntry($uuid);

        return new JsonResponse(null, Response::HTTP_NO_CONTENT);
    }

    /**
     *
     *
     * @throws DocumentManagerException
     */
    #[Route(path: '/api/session/catalog/{uuid}/delete/{filter}', name: 'juweliere_catalog_session_delete_product_filter')]
    public function deleteFilterFromSession(string $uuid, string $filter): RedirectResponse
    {
        $this->filterManager->removeFilter($uuid, $filter);

        $document = $this->documentManager->find($uuid);

        return new RedirectResponse($document->getResourceSegment());
    }

    private function updateFilters(Request $request, StructureInterface $structure, $defaultLimit = 12): void
    {
        $uuid = $structure->getUuid();
        $filters = $request->query->get('filters');
        $page = $request->query->get('page');
        $limit = $request->query->get('limit', $defaultLimit);
        $sort = $request->query->get('sort');
        $dir = $request->query->get('dir', 'asc');

        if ($filters) {
            foreach ($filters as $key => $filter) {
                $data = [
                    'code' => $key,
                    'value' => $filter,
                ];
                $this->filterManager->addFilter($uuid, $data);
            }
        }

        if ($page) {
            $this->filterManager->setPage($uuid, (int) $page);
        }

        if ($limit) {
            $this->filterManager->setLimit($uuid, (int) $limit);
        }

        if ($sort) {
            $this->filterManager->setSorting($uuid, $sort, $dir);
        }
    }

    /**
     * @throws InvalidArgumentException
     */
    protected function bucketLabels(array $attribute, array $buckets): array
    {
        $options = [];

        foreach ($buckets as $bucket) {
            $cachedOption = $this->cache->getItem('option_'.$bucket['key']);

            if ($cachedOption->isHit()) {
                $options[] = [
                    'key' => $bucket['key'],
                    'label' => $cachedOption->get(),
                ];
            } else {
                foreach ($attribute['options'] as $option) {
                    if ($option['code'] === $bucket['key']) {
                        $options[] = [
                            'key' => $bucket['key'],
                            'label' => $option['label'],
                        ];

                        $cachedOption->set($option['label']);
                        $this->cache->save($cachedOption);
                    }
                }
            }
        }

        return $options;
    }
}
