<?php

declare(strict_types=1);

namespace Juweliere\ProductBundle\ElasticSearch;

use Symfony\Component\HttpFoundation\Request;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;

class ElasticProductsParser extends AbstractElasticParser
{
    public function getSingleProduct(string $sku): ?array
    {
        $uri = sprintf('/api/elastic/products/%s', $sku);

        return $this->decoratorClient->get($uri);
    }

    public function getProductArray(array $items): ?array
    {
        $products = [];

        foreach ($items as $item) {
            $product = $this->getSingleProduct($item["product"]["sku"]);

            if ($product && true === $product['isEnabled'] && true === $product['isVisible']) {
                $products[] = $product;
            }
        }

        return $products;
    }

    public function search(string $query, int $page = 1, int $limit = 24, bool $minimal = true): ?array
    {
        $uri = sprintf('/api/elastic/search/products?q=%s&page=%s&limit=%s', $query, $page, $limit);

        if ($minimal) {
            $uri .= '&minimal=true';
        }

        return $this->decoratorClient->get($uri);
    }



    public function suggest(string $query, int $page = 1, int $limit = 24): ?array
    {
        $uri = sprintf('/api/elastic/suggest/products?q=%s&page=%s&limit=%s', $query, $page, $limit);

        return $this->decoratorClient->get($uri);
    }

    public function getRelatedProducts(string $sku, bool $random = false, bool $brandProducts = false): ?array
    {
        $params = ["random" => 0, "brand" => 0];

        if ($random) {
            $params["random"] = 1;
        }

        if ($brandProducts) {
            $params["brand"] = 1;
        }

        $uri = sprintf('/api/elastic/related-products/%s?%s', $sku, http_build_query($params));

        return $this->decoratorClient->get($uri);
    }

    /**
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    public function getProductQuery(Request $request, string $id): ?array
    {
        if ($request->attributes->has('structure')) {
            $uuid = $request->attributes->get('structure')->getUuid();
        } else {
            $uuid = $request->attributes->get('uuid');
        }

        $sessionFilters = $this->catalogFilter->getEntry($uuid);

        if ($sessionFilters) {
            $params = array_merge($sessionFilters, $request->query->all());
        } else {
            $params = $request->query->all();
        }

        $uri = sprintf('/api/elastic/queries/%s?%s', $id, http_build_query($params));

        return $this->decoratorClient->get($uri);
    }

    public function getAll(): ?array
    {
        $uri = '/api/elastic/products?limit=100000';

        return $this->decoratorClient->get($uri);
    }
}
