<?php

declare(strict_types=1);

namespace Juweliere\ProductBundle\Session;

use Symfony\Component\HttpFoundation\RequestStack;

class SessionManager
{
    const string SESSION_LAST_SEEN = "lastseen";

    public function __construct(protected RequestStack $requestStack)
    {
    }

    public function setPage(string $uuid, int $page): void
    {
        $entry = $this->getEntry($uuid);

        $entry['page'] = $page;
        $session = $this->requestStack->getSession();
        $session->set($uuid, $entry);
    }

    public function setLimit(string $uuid, int $limit): void
    {
        $entry = $this->getEntry($uuid);

        $entry['limit'] = $limit;
        $session = $this->requestStack->getSession();
        $session->set($uuid, $entry);
    }

    public function setSorting(string $uuid, string $sorter, string $dir = 'asc'): void
    {
        $entry = $this->getEntry($uuid);

        $entry['sort'] = $sorter;
        $entry['dir'] = $dir;
        $session = $this->requestStack->getSession();
        $session->set($uuid, $entry);
    }

    public function getEntry($uuid): mixed
    {
        $session = $this->requestStack->getSession();

        if ($session->has($uuid)) {
            return $session->get($uuid, []);
        }

        return null;
    }

    public function getLastSeenProducts(array $product, int $limit = 4): array
    {
        $lastSeenProducts = [];
        $newLastSeenProducts = [];
        $count = 0;

        /** @var array $attributes */
        $attributes = ["sku", "reference", "qty", "image", "brand", "name", "showPrice", "rrp"];

        $session = $this->requestStack->getSession();
        if ($session->has(self::SESSION_LAST_SEEN)) {
            $lastSeenProducts = $session->get(self::SESSION_LAST_SEEN);
        }

        foreach ($lastSeenProducts as $lastSeenProduct) {
            if ($count > $limit) {
                break;
            }

            if ($lastSeenProduct["sku"] === $product["sku"]) {
                continue;
            }

            $newLastSeenProducts[] = $lastSeenProduct;
            $count++;
        }

        $productDataFiltered = array_intersect_key($product, array_flip($attributes));

        array_unshift($newLastSeenProducts, $productDataFiltered);

        $session->set(self::SESSION_LAST_SEEN, $newLastSeenProducts);
        $session->save();

        array_shift($newLastSeenProducts);

        return $newLastSeenProducts;
    }
}
