<?php

declare(strict_types=1);

namespace Juweliere\ProductBundle\Controller\Website;

use GuzzleHttp\Exception\GuzzleException;
use Psr\Cache\InvalidArgumentException;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Juweliere\ProductBundle\ElasticSearch\ElasticProductsParser;
use Massive\Bundle\SearchBundle\Search\SearchManagerInterface;
use Sulu\Bundle\MediaBundle\Media\Manager\MediaManagerInterface;
use Sulu\Component\Webspace\Analyzer\RequestAnalyzerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Cache\Adapter\FilesystemAdapter;
use Symfony\Component\Cache\Adapter\RedisAdapter;
use Symfony\Component\Cache\CacheItem;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\KernelInterface;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;

class SearchController extends AbstractController
{
    const string ROLEX_SEARCH_CACHE_KEY = "search.rolex";


    public function __construct(private readonly HttpClientInterface $rolexClient, private readonly MediaManagerInterface $mediaManager, private readonly ParameterBagInterface $parameterBag)
    {
    }

    /**
     *
     *
     *
     * @throws TransportExceptionInterface
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     */
    #[Route(path: '%juweliere_product.search_uri%', name: 'search', options: ['expose' => true])]
    public function search(
        Request $request,
        ElasticProductsParser $productsParser,
        RequestAnalyzerInterface $requestAnalyzer,
        SearchManagerInterface $searchManager,
        KernelInterface $kernel
    ): Response
    {
        /** @var string|null $query */
        $query = $request->query->get('q');
        $page = (int) $request->query->get('page', 1);
        $limit = (int) $request->query->get('limit', 24);

        if (!$query) {
            return $this->render('pages/search.html.twig', [
            ]);
        }

        /**
         * Special Rolex handling
         */
        if ($this->isRolexSearch($query)) {
            return $this->redirect("/rolex");
        }

        /**
         * Special Wellendorff handling
         */
        if ($this->isWellendorf($query)) {
            return $this->redirect("/wellendorff");
        }

        $locale = $requestAnalyzer->getCurrentLocalization()->getLocale();
        $webspace = $requestAnalyzer->getWebspace();

        $queryString = '';
        if (\strlen($query) < 3) {
            $queryString .= '+("' . self::escapeDoubleQuotes($query) . '") ';
        } else {
            $queryValues = \explode(' ', $query);
            foreach ($queryValues as $queryValue) {
                if (\strlen($queryValue) > 2) {
                    $queryString .= '+("' . self::escapeDoubleQuotes($queryValue) . '" OR ' .
                        \preg_replace('/([^\pL\s\d])/u', '?', $queryValue) . '* OR ' .
                        \preg_replace('/([^\pL\s\d])/u', '', $queryValue) . '~) ';
                } else {
                    $queryString .= '+("' . self::escapeDoubleQuotes($queryValue) . '") ';
                }
            }
        }

        /** @var array $indexes */
        $indexes = explode(",", $this->parameterBag->get('search_indexes'));

        $pages = [];

        try {
            $pages = $searchManager
                ->createSearch($queryString)
                ->locale($locale)
                ->indexes(
                    \str_replace(
                        '#webspace#',
                        $webspace->getKey(),
                        $indexes
                    )
                )
                ->setLimit(128)
                ->execute();
        } catch (\Throwable $throwable) {
            if ($kernel->getEnvironment() === "dev") {
//                var_dump($e->getMessage());
//                die();
            }
        }

        /*echo "<pre>";
        foreach ($pages as $hit) {
            var_dump($hit->getDocument()->getTitle());
            var_dump($hit->getDocument()->getDescription());
            var_dump($hit->getDocument()->getUrl());
            var_dump($hit->getDocument()->getImageUrl());
            var_dump($hit->getDocument()->getIndex());
            #var_dump($hit->getDocument()->getFields());
            #var_dump($hit->getScore());
            #var_dump($hit);
        }
        echo "</pre>";
        die();*/

        #echo "<pre>";
//        foreach ($pages as $hit) {
        #var_dump($hit->getDocument()->getTitle());

        #var_dump($hit->getDocument()->getDescription());
        #var_dump($hit->getDocument()->getUrl());
        #var_dump($hit->getDocument()->getImageUrl());
        #var_dump($hit->getDocument()->getIndex());
        #var_dump($hit->getDocument()->getFields());
        #var_dump($hit->getScore());
        #var_dump($hit);
//        }
#echo "</pre>";

        foreach ($pages as $hit) {
            try {

                $excerptImag = json_decode((string) $hit->getDocument()->getFields()["excerptIcon"]->getValue());

                if ($hit->getDocument()->getFields()["_structure_type"]->getValue() == "brand" &&
                    !empty($hit->getDocument()->getImageUrl())
                ) {
                    $hit->getDocument()->setImageUrl(str_replace("sulu-100x100", "100x", $hit->getDocument()->getImageUrl()));
                } elseif (!empty($excerptImag->ids)) {
                    $hit->getDocument()->setImageUrl($excerptImag->ids[0]);
                }

                if (!empty(trim((string) $hit->getDocument()->getFields()["excerptTitle"]->getValue()))) {
                    $hit->getDocument()->setTitle(trim((string) $hit->getDocument()->getFields()["excerptTitle"]->getValue()));
                }

                if (!empty(trim((string) $hit->getDocument()->getFields()["excerptDescription"]->getValue()))) {
                    $hit->getDocument()->setDescription(html_entity_decode(strip_tags(trim((string) $hit->getDocument()->getFields()["excerptDescription"]->getValue()))));
                }

                foreach($hit->getDocument()->getFields() as $field) {
                    if(preg_match('/_text$/', (string) $field->getName()) && preg_match('/'.strtolower($query).'/', strtolower((string) $field->getValue())) && strlen($query) < strlen((string) $field->getValue())){
                        $hit->getDocument()->setDescription(html_entity_decode(strip_tags(trim((string) $field->getValue()))));
                        break;
                    }
                }

                if (!empty($hit->getDocument()->getImageUrl())) {
                    $imgUrl = $this->mediaManager->getById($hit->getDocument()->getImageUrl(), "de")->getFormats()["sulu-100x100"];
                    $hit->getDocument()->setImageUrl($imgUrl);
                }
            } catch (\Throwable $e) {
                if ($kernel->getEnvironment() === "dev") {
//                    var_dump($e->getMessage());
//                    die();
                }
            }
        }

        /** @var array $products */
        $products = $productsParser->search($query, $page, $limit);

        $response = $this->render('pages/search.html.twig', [
            'products' => $products,
            'pages' => $pages,
            'query' => $query
        ]);

        $response->setSharedMaxAge(86400);

        $response->headers->addCacheControlDirective('must-revalidate', true);

        return $response;
    }

    /**
     * @param $query
     * @throws GuzzleException
     * @throws InvalidArgumentException
     */
    private function isRolexSearch($query): bool {
        /** @var RedisAdapter $cache */
        $cache = new FilesystemAdapter();

        /** @var CacheItem $rolexSearch */
        $rolexSearch = $cache->getItem(self::ROLEX_SEARCH_CACHE_KEY);

        if (!$rolexSearch->isHit())
        {
            $rolexSearch->expiresAfter(3600);

            /** @var Client $client */
            $client = new Client();

            try {
                $result = $this->rolexClient->request('GET', '/api/keywords.json');
            } catch (RequestException) {
                return false;
            }

            /** @var object $rolexSearches */
            $rolexSearches = json_decode($result->getContent());

            foreach ($rolexSearches as $item) {
                $keywords[] = $item->keyword;
            }

            /** @var string $rolexSearchString */
            $rolexSearchString = implode("|", $keywords);

            $rolexSearch->set($rolexSearchString);
            $cache->save($rolexSearch);
        }

        if (preg_match("/(" . $rolexSearch->get() . ")/i", (string) $query)) {
            return true;
        };

        return false;
    }

    /**
     * @param $query
     */
    private function isWellendorf($query): bool
    {
        if (preg_match("/wellendorff/", strtolower((string) $query))) {
            return true;
        }

        return false;
    }

    /**
     * Returns the string with escaped quotes.
     *
     * @param string $query
     */
    private static function escapeDoubleQuotes($query): string
    {
        return \str_replace('"', '\\"', $query);
    }
}
