<?php

declare(strict_types=1);

namespace Juweliere\ProductBundle\Controller\Website;

use Psr\Cache\InvalidArgumentException;
use Psr\Log\LoggerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpClient\HttpClient;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Psr\Cache\CacheItemPoolInterface;

class FilterController extends AbstractController
{
    private readonly string $decoratorBaseUri;

    public function __construct(
        private readonly CacheItemPoolInterface         $cache,
        private readonly LoggerInterface                $logger
    )
    {
        $this->decoratorBaseUri = $this->getParameter('decorator_base_uri');
    }

    /**
     * @throws ClientExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws InvalidArgumentException
     * @throws RedirectionExceptionInterface
     * @throws ServerExceptionInterface
     * @throws TransportExceptionInterface
     */
    #[Route(path: '/attributes/{code}/name', name: 'juweliere_catalog_attribute_name')]
    public function attribute(Request $request, string $code): Response
    {
        $uuid = $request->get('uuid');
        $cachedAttribute = $this->cache->getItem('filter_'.$code);
        $attribute = null;

        if ($cachedAttribute->isHit()) {
            $this->logger->info("Attribute cache hit for code: " . $code);
            $attribute = $cachedAttribute->get();

            $response = $this->render(
                'pages/product/filter.html.twig',
                [
                    'attribute' => $attribute,
                    'options' => $this->bucketLabels($attribute, $request->get('buckets')),
                    'uuid' => $uuid,
                ]
            );

            $response->setPublic();
            $response->setMaxAge(3600);

            return $response;
        }

        $client = HttpClient::create();
        $attributeResponse = $client->request('GET', sprintf('/api/elastic/attributes/%s', $code), [
            'base_uri' => $this->decoratorBaseUri,
        ]);

        if ($attributeResponse->getStatusCode() < 400) {
            $attribute = $attributeResponse->toArray(false);
            $cachedAttribute->set($attribute);
            $this->cache->save($cachedAttribute);

            $this->logger->info("Save new attribute for code: " . $code);
        }

        $response = $this->render(
            'pages/product/filter.html.twig',
            [
                'attribute' => $attribute,
                'options' => $this->bucketLabels($attribute, $request->get('buckets')),
                'uuid' => $uuid,
            ]
        );

        $response->setPublic();
        $response->setMaxAge(60);

        return $response;
    }

    #[Route(path: '/filters/{uuid}', name: 'juweliere_catalog_session_filters')]
    public function filters(Request $request, string $uuid): Response
    {
        $session = $request->getSession();
        if ($session->has($uuid) && isset($session->get($uuid)['filters'])) {
            $data = [];
            foreach ($session->get($uuid)['filters'] as $key => $filter) {

                $cachedAttribute = $this->cache->getItem('filter_'.$key);
                $attribute = null;

                if ($cachedAttribute->isHit()) {
                    $this->logger->error("Attribute filters cache hit for key: " . $key);
                    $attribute = $cachedAttribute->get();
                }

                if ($attribute) {
                    $label = null;
                    foreach ($attribute['options'] as $option) {
                        if ($option['code'] === $filter) {
                            $label = $option['label'];
                        }
                    }

                    $data[] = [
                        'attribute' => [
                            'key' => $key,
                            'value' => $attribute['name'],
                        ],
                        'option' => [
                            'key' => $filter,
                            'value' => $label,
                        ],
                    ];
                }
            }

            $response = $this->render(
                'pages/catalog/_selection.html.twig',
                [
                    'selectedFilters' => $data,
                    'uuid' => $uuid,
                ]
            );

            $response->setPublic();
            $response->setMaxAge(3600);

            return $response;
        }

        return new Response();
    }

    /**
     *
     * @throws InvalidArgumentException
     */
    protected function bucketLabels(array $attribute, array $buckets): array
    {
        $options = [];

        foreach ($buckets as $bucket) {
            $cachedOption = $this->cache->getItem('option_'.$bucket['key']);

            if ($cachedOption->isHit()) {
                $this->logger->error("Filter bucket label cache hit for key: " . $bucket['key']);
                $options[] = [
                    'key' => $bucket['key'],
                    'label' => $cachedOption->get(),
                ];
            } else {
                foreach ($attribute['options'] as $option) {
                    if ($option['code'] === $bucket['key']) {
                        $options[] = [
                            'key' => $bucket['key'],
                            'label' => $option['label'],
                        ];

                        $cachedOption->set($option['label']);
                        $this->cache->save($cachedOption);
                    }
                }
            }
        }

        return $options;
    }
}
