<?php

declare(strict_types=1);

namespace Juweliere\UserBundle\Controller;

use Juweliere\JuwApiClient\Entity\User\WishlistItem;
use Juweliere\ProductBundle\Entity\Product;
use Juweliere\ProductBundle\Repository\ProductRepository;
use Juweliere\UserBundle\Entity\User;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;

/**
 * @Route("/user", name="juweliere_user_user_")
 */
class WishlistController extends BaseController
{
    /**
     * @Route("/wishlist", name="wishlist")
     * @param ProductRepository $productRepository
     * @return Response
     */
    public function wishlist(ProductRepository $productRepository): Response
    {
        /** @var WishlistItem[]|null $wishlist */
        $wishlistItems = $this->getUser()->getWishlistItems();
        $products = [];

        if ($wishlistItems) {
            /** @var WishlistItem $wishlistItem */
            foreach ($wishlistItems as $wishlistItem) {
                $product = $productRepository->findOneBy(['sku' => $wishlistItem->getSku()]);
                $products[] = $product;
            }
        }

        return $this->render('@JuweliereUser/pages/user/wishlist.html.twig', [
            'products' => $products,
        ]);
    }

    /**
     * @Route("/wishlist/remove", name="wishlist_add_xhr")
     * @return Response
     */
    public function removeFromWishlistXhr(Request $request): Response
    {
        /** @var User $user */
        $user = $this->getUser();

        $data = json_decode($request->getContent(), true);

        $wishlistItem = new WishlistItem();
        $wishlistItem->setSku($data["sku"]);
        $wishlistItem->setUser($user->getId());

        try {
            $this->juwApi->wishlistItems->addItemToWishlist($wishlistItem);
            $this->sessionRefreshUser();
        } catch (\Throwable $e) {
            var_dump($e->getMessage());
            die();
        }

        return new JsonResponse(array("result" => true));
    }

    /**
     * @Route("/wishlist/add", name="wishlist_add_xhr")
     * @return Response
     */
    public function addToWishlistXhr(Request $request): Response
    {
        /** @var User $user */
        $user = $this->getUser();

        $data = json_decode($request->getContent(), true);

        $wishlistItem = new WishlistItem();
        $wishlistItem->setSku($data["sku"]);
        $wishlistItem->setUser($user->getId());

        try {
            $this->juwApi->wishlistItems->addItemToWishlist($wishlistItem);
            $this->sessionRefreshUser();
        } catch (\Throwable $e) {
            var_dump($e->getMessage());
            die();
        }

        return new JsonResponse(array("result" => true));
    }

    /**
     * @Route("/wishlist/{sku}/add", name="wishlist_add")
     * @return Response
     */
    public function addToWishlist($sku): Response
    {
        /** @var User $user */
        $user = $this->getUser();

        $wishlistItem = new WishlistItem();
        $wishlistItem->setSku($sku);
        $wishlistItem->setUser($user->getId());

        try {
            $this->juwApi->wishlistItems->addItemToWishlist($wishlistItem);
            $this->sessionRefreshUser();
        } catch (\Throwable $e) {
            var_dump($e->getMessage());
            die();
        }

        return $this->redirectToRoute('juweliere_user_user_wishlist');
    }

    /**
     * @Route("/wishlist/{sku}/remove", name="wishlist_remove")
     * @param Product $product
     * @return Response
     */
    public function removeFromWishlist(Request $request, $sku): Response
    {
        try {
        $wishlistItems = $this->getUser()->getWishlistItems();

        if ($wishlistItems) {
            /** @var WishlistItem $wishlistItem */
            foreach ($wishlistItems as $wishlistItem) {
                if ($wishlistItem->getSku() == $sku) {
                    try {
                        $this->juwApi->wishlistItems->deleteItemFromWishlist($wishlistItem->getId());
                        $this->sessionRefreshUser();
                    } catch (\Throwable $e) {
                        var_dump($e->getMessage());
                        die();
                    }
                }
            }
        }
        } catch (\Throwable $e) {
            var_dump($e->getMessage());
        }

        if ($request->isXmlHttpRequest()) {
            return new JsonResponse(array("result" => true));
        } else {
            return $this->redirectToRoute('juweliere_user_user_wishlist');
        }
    }
}
