<?php

namespace Juweliere\UserBundle\Security;

use Doctrine\ORM\EntityManagerInterface;
use GuzzleHttp\Exception\ClientException;
use Juweliere\ApiBundle\Service\JuwApi;
use Juweliere\JuwApiClient\Entity\User;
use Juweliere\UserBundle\Controller\LoginController;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Encoder\UserPasswordEncoderInterface;
use Symfony\Component\Security\Core\Exception\CustomUserMessageAuthenticationException;
use Symfony\Component\Security\Core\Exception\InvalidCsrfTokenException;
use Symfony\Component\Security\Core\Security;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Security\Core\User\UserProviderInterface;
use Symfony\Component\Security\Csrf\CsrfToken;
use Symfony\Component\Security\Csrf\CsrfTokenManagerInterface;
use Symfony\Component\Security\Guard\Authenticator\AbstractFormLoginAuthenticator;
use Symfony\Component\Security\Guard\PasswordAuthenticatedInterface;
use Symfony\Component\Security\Core\User\PasswordAuthenticatedUserInterface;
use Symfony\Component\Security\Http\Util\TargetPathTrait;

class LoginFormAuthenticator extends AbstractFormLoginAuthenticator implements PasswordAuthenticatedUserInterface
{
    use TargetPathTrait;

    public const LOGIN_ROUTE = 'juweliere_user_user_login';

    private $entityManager;
    private $urlGenerator;
    private $csrfTokenManager;
    private $passwordEncoder;
    private JuwApi $juwApi;
    private LoggerInterface $logger;
    private SessionInterface $session;

    /**
     * LoginFormAuthenticator constructor.
     * @param EntityManagerInterface $entityManager
     * @param UrlGeneratorInterface $urlGenerator
     * @param CsrfTokenManagerInterface $csrfTokenManager
     * @param UserPasswordEncoderInterface $passwordEncoder
     * @param JuwApi $juwApi
     * @param LoggerInterface $logger
     * @param SessionInterface $session
     */
    public function __construct(
        EntityManagerInterface $entityManager,
        UrlGeneratorInterface $urlGenerator,
        CsrfTokenManagerInterface $csrfTokenManager,
        UserPasswordEncoderInterface $passwordEncoder,
        JuwApi $juwApi,
        LoggerInterface $logger,
        SessionInterface $session
    )
    {
        $this->entityManager = $entityManager;
        $this->urlGenerator = $urlGenerator;
        $this->csrfTokenManager = $csrfTokenManager;
        $this->passwordEncoder = $passwordEncoder;
        $this->juwApi = $juwApi;
        $this->logger = $logger;
        $this->session = $session;
    }

    public function supports(Request $request)
    {
        return self::LOGIN_ROUTE === $request->attributes->get('_route')
            && $request->isMethod('POST');
    }

    public function getCredentials(Request $request)
    {
        $credentials = [
            'username' => $request->request->get('username'),
            'password' => $request->request->get('password'),
            'csrf_token' => $request->request->get('_csrf_token'),
        ];
        $request->getSession()->set(
            Security::LAST_USERNAME,
            $credentials['username']
        );

        return $credentials;
    }

    public function getUser($credentials, UserProviderInterface $userProvider)
    {
        $token = new CsrfToken('authenticate', $credentials['csrf_token']);

        if (!$this->csrfTokenManager->isTokenValid($token)) {
            throw new InvalidCsrfTokenException();
        }

        $login = new User\Login();
        $login->setUsername( $credentials['username']);
        $login->setPassword( $credentials['password']);

        try {
            $userApi = $this->juwApi->users->login($login);
        } catch (ClientException $e) {
            throw new CustomUserMessageAuthenticationException('Invalid credentials.');
        } catch (\Throwable $e) {
            $this->logger->critical("Login Form Error: " . $e->getMessage());
            throw new CustomUserMessageAuthenticationException('Internal error.');
        }

        return $this->mapApiUser($userApi);
    }

    /**
     * @param User $userApi
     * @return \Juweliere\UserBundle\Entity\User
     */
    public function mapApiUser(User $userApi)
    {
        $user = new \Juweliere\UserBundle\Entity\User();

        $user->setId($userApi->getId());
        $user->setEmail($userApi->getEmail());
        $user->setGender($userApi->getGender());
        $user->setFirstName($userApi->getFirstName());
        $user->setLastName($userApi->getLastName());
        $user->setAddresses($userApi->getAddresses());
        $user->setWishlistItems($userApi->getWishlistItems());
        $user->setContactPhone($userApi->getContactPhone());
        $user->setContactEmail($userApi->getContactEmail());
        $user->setBirthdate($userApi->getBirthdate());
        $user->setPhone($userApi->getPhone());

        return $user;
    }

    public function checkCredentials($credentials, UserInterface $user)
    {
        return true;
    }

    /**
     * Used to upgrade (rehash) the user's password automatically over time.
     */
    public function getPassword(): ?string
    {
        return "";
    }

    public function onAuthenticationSuccess(Request $request, TokenInterface $token, $providerKey)
    {
        if ($redirect = $this->session->get(LoginController::SESSION_LOGIN_REDIRECT)) {
            return new RedirectResponse($redirect);
        }

        if ($targetPath = $this->getTargetPath($request->getSession(), $providerKey)) {
            return new RedirectResponse($targetPath);
        }

        return new RedirectResponse($this->urlGenerator->generate('juweliere_user_user_profile_edit'));
    }

    protected function getLoginUrl()
    {
        return $this->urlGenerator->generate(self::LOGIN_ROUTE);
    }
}
