<?php

declare(strict_types=1);

namespace Juweliere\UserBundle\Security;

use GuzzleHttp\Exception\ClientException;
use Juweliere\ApiBundle\Service\JuwApi;
use Juweliere\JuwApiClient\Entity\User;
use Juweliere\UserBundle\Controller\LoginController;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Core\Exception\CustomUserMessageAuthenticationException;
use Symfony\Component\Security\Core\Exception\InvalidCsrfTokenException;
use Symfony\Component\Security\Csrf\CsrfToken;
use Symfony\Component\Security\Csrf\CsrfTokenManagerInterface;
use Symfony\Component\Security\Http\Authentication\AuthenticationFailureHandlerInterface;
use Symfony\Component\Security\Http\Authentication\AuthenticationSuccessHandlerInterface;
use Symfony\Component\Security\Http\Authenticator\FormLoginAuthenticator;
use Symfony\Component\Security\Http\Authenticator\Passport\Badge\RememberMeBadge;
use Symfony\Component\Security\Http\Authenticator\Passport\Badge\UserBadge;
use Symfony\Component\Security\Http\Authenticator\Passport\Passport;
use Symfony\Component\Security\Http\Authenticator\Passport\SelfValidatingPassport;
use Symfony\Component\Security\Http\HttpUtils;
use Symfony\Component\Security\Http\Util\TargetPathTrait;

class LoginFormAuthenticator extends FormLoginAuthenticator
{
    use TargetPathTrait;

    public const LOGIN_ROUTE = 'juweliere_user_user_login';

    private UrlGeneratorInterface $urlGenerator;
    private CsrfTokenManagerInterface $csrfTokenManager;
    private JuwApi $juwApi;
    private LoggerInterface $logger;
    private SessionInterface $session;
    private UserProvider $userProvider;

    /**
     * @param HttpUtils $httpUtils
     * @param AuthenticationSuccessHandlerInterface $successHandler
     * @param AuthenticationFailureHandlerInterface $failureHandler
     * @param UserProvider $userProvider
     * @param UrlGeneratorInterface $urlGenerator
     * @param JuwApi $juwApi
     * @param LoggerInterface $logger
     * @param RequestStack $requestStack
     * @param CsrfTokenManagerInterface $csrfTokenManager
     */
    public function __construct(
        HttpUtils $httpUtils,
        AuthenticationSuccessHandlerInterface $successHandler,
        AuthenticationFailureHandlerInterface $failureHandler,
        UserProvider $userProvider,
        UrlGeneratorInterface $urlGenerator,
        JuwApi $juwApi,
        LoggerInterface $logger,
        RequestStack $requestStack,
        CsrfTokenManagerInterface $csrfTokenManager
    )
    {
        $options = array(
            "use_forward" => false,
            "username_parameter" => "username",
            "username_password" => "password"
        );

        parent::__construct($httpUtils, $userProvider, $successHandler, $failureHandler, $options);

        $this->urlGenerator = $urlGenerator;
        $this->juwApi = $juwApi;
        $this->logger = $logger;
        $this->session = $requestStack->getSession();
        $this->csrfTokenManager = $csrfTokenManager;
        $this->userProvider = $userProvider;
    }

    public function supports(Request $request): bool
    {
        return self::LOGIN_ROUTE === $request->attributes->get('_route')
            && $request->isMethod('POST');
    }

    public function authenticate(Request $request): Passport
    {
        $token = new CsrfToken('authenticate', $request->get('_csrf_token'));

        if (!$this->csrfTokenManager->isTokenValid($token)) {
            throw new InvalidCsrfTokenException();
        }

        $login = new User\Login();
        $login->setUsername( $request->get('username') );
        $login->setPassword( $request->get('password'));

        try {
            $userApi = $this->juwApi->users->login($login);
        } catch (ClientException $e) {
            throw new CustomUserMessageAuthenticationException('Invalid credentials: ' . $e->getCode());
        } catch (\Throwable $e) {
            $this->logger->critical("Login Form Error: " . $e->getMessage());
            throw new CustomUserMessageAuthenticationException('Internal error.');
        }

        $user = $this->userProvider->mapApiUser($userApi);

        return new SelfValidatingPassport(
            new UserBadge($user->getId(), [$this->userProvider, "loadUserByIdentifier"]),
            [new RememberMeBadge()]
        );
    }

    public function onAuthenticationSuccess(Request $request, TokenInterface $token, $firewallName): RedirectResponse
    {
        if ($redirect = $this->session->get(LoginController::SESSION_LOGIN_REDIRECT)) {
            return new RedirectResponse($redirect);
        }

        if ($targetPath = $this->getTargetPath($request->getSession(), $firewallName)) {
            return new RedirectResponse($targetPath);
        }

        return new RedirectResponse($this->urlGenerator->generate('juweliere_user_user_profile_edit'));
    }

    public function onAuthenticationFailure(Request $request, AuthenticationException $exception): Response
    {
        return new RedirectResponse($this->urlGenerator->generate(self::LOGIN_ROUTE));
    }

    protected function getLoginUrl(Request $request): string
    {
        return $this->urlGenerator->generate(self::LOGIN_ROUTE);
    }
}
