<?php

declare(strict_types=1);


namespace Juweliere\UserBundle\Security;

use Exception;
use GuzzleHttp\Exception\GuzzleException;
use Juweliere\ApiBundle\Service\JuwApi;
use Juweliere\UserBundle\Entity\User;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\Security\Core\Exception\UnsupportedUserException;
use Symfony\Component\Security\Core\User\PasswordUpgraderInterface;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Security\Core\User\UserProviderInterface;

class UserProvider implements UserProviderInterface, PasswordUpgraderInterface
{
    const SESSION_KEY_REFRESH_USER = "refreshUser";

    private JuwApi $juwApi;
    private Session $session;
    private ?User $user = null;

    /**
     * UserProvider constructor.
     * @param JuwApi $juwApi
     * @param SessionInterface $session
     */
    public function __construct(JuwApi $juwApi, SessionInterface $session)
    {
        $this->juwApi = $juwApi;
        $this->session = $session;
    }

    /**
     * The loadUserByIdentifier() method was introduced in Symfony 5.3.
     * In previous versions it was called loadUserByUsername()
     *
     * Symfony calls this method if you use features like switch_user
     * or remember_me. If you're not using these features, you do not
     * need to implement this method.
     *
     * @param string $identifier
     * @return UserInterface
     * @throws Exception
     */
    public function loadUserByIdentifier(string $identifier): UserInterface
    {
        if (is_null($this->user)) {
            $apiUser = $this->juwApi->users->get($identifier);

            return $this->mapApiUser($apiUser);
        }

        return $this->user;
    }

    /**
     * Refreshes the user after being reloaded from the session.
     *
     * When a user is logged in, at the beginning of each request, the
     * User object is loaded from the session and then this method is
     * called. Your job is to make sure the user's data is still fresh by,
     * for example, re-querying for fresh User data.
     *
     * If your firewall is "stateless: true" (for a pure API), this
     * method is not called.
     *
     * @param UserInterface $user
     * @return UserInterface
     * @throws GuzzleException
     */
    public function refreshUser(UserInterface $user): UserInterface
    {
        if (!$user instanceof User) {
            throw new UnsupportedUserException(sprintf('Invalid user class "%s".', get_class($user)));
        }

        if ($this->session->get(self::SESSION_KEY_REFRESH_USER) === 1) {

            $apiUser = $this->juwApi->users->get($user->getId());
            $this->session->remove(self::SESSION_KEY_REFRESH_USER);

            return $this->mapApiUser($apiUser);
        }

        return $user;
    }

    /**
     * @param \Juweliere\JuwApiClient\Entity\User $userApi
     * @return \Juweliere\UserBundle\Entity\User
     */
    public function mapApiUser(\Juweliere\JuwApiClient\Entity\User $userApi)
    {
        $this->user = new \Juweliere\UserBundle\Entity\User();

        $this->user->setId($userApi->getId());
        $this->user->setEmail($userApi->getEmail());
        $this->user->setGender($userApi->getGender());
        $this->user->setFirstName($userApi->getFirstName());
        $this->user->setLastName($userApi->getLastName());
        $this->user->setAddresses($userApi->getAddresses());
        $this->user->setWishlistItems($userApi->getWishlistItems());
        $this->user->setContactPhone($userApi->getContactPhone());
        $this->user->setContactEmail($userApi->getContactEmail());
        $this->user->setBirthdate($userApi->getBirthdate());
        $this->user->setPhone($userApi->getPhone());

        return $this->user;
    }

    /**
     * Tells Symfony to use this provider for this User class.
     */
    public function supportsClass($class): bool
    {
        return User::class === $class || is_subclass_of($class, User::class);
    }

    /**
     * Upgrades the hashed password of a user, typically for using a better hash algorithm.
     */
    public function upgradePassword(UserInterface $user, string $newHashedPassword): void
    {
        // TODO: when hashed passwords are in use, this method should:
        // 1. persist the new password in the user storage
        // 2. update the $user object with $user->setPassword($newHashedPassword);
    }

    public function loadUserByUsername($username)
    {
        // TODO: Implement loadUserByUsername() method.
    }

    public function setUser(User $user) {
        $this->user = $user;
    }
}
