<?php
declare(strict_types=1);

namespace Juweliere\UserBundle\Controller;

use Juweliere\JuwApiClient\Entity\User\Registration;
use Juweliere\UserBundle\Entity\User;
use Juweliere\UserBundle\Form\RegistrationFormType;
use Juweliere\UserBundle\Security\LoginFormAuthenticator;
use Juweliere\UserBundle\Security\UserProvider;
use Psr\Log\LoggerInterface;
use Symfony\Bridge\Twig\Mime\TemplatedEmail;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Component\Mime\Address;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Core\Authentication\Token\UsernamePasswordToken;
use Symfony\Component\Security\Core\Encoder\UserPasswordEncoderInterface;
use Symfony\Component\Security\Core\User\UserProviderInterface;
use SymfonyCasts\Bundle\ResetPassword\Controller\ResetPasswordControllerTrait;
use SymfonyCasts\Bundle\ResetPassword\Exception\ResetPasswordExceptionInterface;
use SymfonyCasts\Bundle\ResetPassword\ResetPasswordHelperInterface;

/**
 * @Route("/user", name="juweliere_user_user_")
 */
class RegistrationController extends BaseController
{

    /**
     * Display & process form to request a password reset.
     *
     * @Route("/register", name="register")
     */
    public function request(Request $request, UserProvider $userProvider): Response
    {
        $newUser = new Registration();

        $form = $this->createForm(RegistrationFormType::class, $newUser);
        $form->handleRequest($request);

        if (!empty($newUser->getEmailRepeat())) {
            $this->logger->error("Registration honeypot: " . json_encode($request->request->all()));
            return $this->redirectToRoute('juweliere_user_user_register_success');
        }

        $error = false;

        if ($form->isSubmitted() && $form->isValid()) {

            if (!$this->checkSt($request->request->get("st"), $newUser->getEmail())) {
                $this->logger->error("Registration honeypot (submit time): " . json_encode($request->request->all()));
                return $this->redirectToRoute('juweliere_user_user_register_success');
            }

            try {

                $userApi = $this->juwApi->users->create($newUser);

                $user = $userProvider->mapApiUser($userApi);

                $token = new UsernamePasswordToken($user, null, 'main', $user->getRoles());

                $this->get('security.token_storage')->setToken($token);
                $this->get('session')->set('_security_main', serialize($token));
                $this->get('session')->set('_new_user', true);

                return $this->redirectToRoute('juweliere_user_user_register_success');

            } catch (\Throwable $e) {
                $error = true;
            }

        }

        $template = '@JuweliereUser/pages/user/register.html.twig';

        $response = $this->render(
            $template,
            [
                'registrationForm' => $form->createView(),
                'error' => $error
            ]);

        return $response;
    }



    /**
     * Display & process form to request a password reset.
     *
     * @Route("/register/success", name="register_success")
     */
    public function success(Request $request): Response
    {
        if ($redirect = $this->session->get(LoginController::SESSION_LOGIN_REDIRECT)) {
            return new RedirectResponse($redirect);
        }

        #return $this->redirectToRoute('juweliere_user_user_profile_edit');
        $template = '@JuweliereUser/pages/user/profile.html.twig';

        $newUser = $this->get('session')->get('_new_user');

        if ($newUser) {
            $this->get('session')->set('_new_user', false);
            $response = $this->render(
                $template,
                [
                    'user' => $this->getUser(),
                    'newUser' => $newUser
                ]);

            return $response;
        }

        return $this->redirectToRoute('juweliere_user_user_profile');

    }

    private function checkSt($st, $email)
    {
        if (!empty($st) && strlen($st) === 3) {
            $checkSt = substr($email, 1, 1) . substr($email, -2, 1);
            if ($checkSt === substr($st, 0, 2) && intval(substr($st, 2) > 3)) {
                return true;
            }
        }

        return false;
    }
}
