<?php
declare(strict_types=1);

namespace Juweliere\UserBundle\Controller;

use GuzzleHttp\Exception\ClientException;
use Juweliere\CheckoutBundle\Constants;
use Juweliere\UserBundle\Form\AddressFormType;
use Juweliere\UserBundle\Form\CancelOrderFormType;
use Juweliere\UserBundle\Form\ReturnAnnouncedFormType;
use Juweliere\UserBundle\Security\UserProvider;
use Symfony\Component\Security\Core\Security;

use App\Entity\User;
use App\Form\ChangePasswordFormType;
use App\Form\ResetPasswordRequestFormType;
use Juweliere\UserBundle\Form\RegistrationFormType;
use Symfony\Bridge\Twig\Mime\TemplatedEmail;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Component\Mime\Address;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Core\Encoder\UserPasswordEncoderInterface;
use SymfonyCasts\Bundle\ResetPassword\Controller\ResetPasswordControllerTrait;
use SymfonyCasts\Bundle\ResetPassword\Exception\ResetPasswordExceptionInterface;
use SymfonyCasts\Bundle\ResetPassword\ResetPasswordHelperInterface;

/**
 * @Route("/user", name="juweliere_user_user_")
 */
class OrdersController extends BaseController
{

    /**
     * Display & process form to request a password reset.
     *
     * @Route("/orders", name="orders")
     */
    public function list(Request $request): Response
    {
        $template = '@JuweliereUser/pages/user/orders.html.twig';

        $displayOrders = [];
        $hideOrderStatus = array("NEW", "INIT");

        /** @var \Juweliere\JuwApiClient\Entity\User $orderUser */
        try {
            $orders = $this->juwApi->orders->getByUser($this->getUser()->getId())->getOrders();
            foreach ($orders as $order) {
                if (!in_array($order->getStatus(), $hideOrderStatus)) {
                    $order->setStatus($this->mapStatusLabel($order->getStatus()));
                    $displayOrders[] = $order;
                }
            }
        } catch (ClientException $e) {

        }

        $response = $this->render(
            $template,
            [
                'orders' => $displayOrders,
                'user' => $this->getUser()
            ]);

        return $response;
    }

    /**
     * Display & process form to request a password reset.
     *
     * @Route("/orders/{id}", name="orders_detail")
     */
    public function new(Request $request, $id): Response
    {
        $newAddress=  new \Juweliere\JuwApiClient\Entity\Address();
        $form = $this->createForm(AddressFormType::class, $newAddress);
        $form->handleRequest($request);

        $order = $this->juwApi->orders->get($id);
        $orderStatusCode = $order->getStatus();

        $order->setStatus($this->mapStatusLabel($order->getStatus()));

        $template = '@JuweliereUser/pages/user/orders/detail.html.twig';

        $response = $this->render(
            $template,
            [
                'orderStatusCode' => $orderStatusCode,
                'order' => $order,
                'user' => $this->getUser(),
                'countries' => $this->getCountries()
            ]);

        return $response;
    }

    /**
     * Display & process form to request a password reset.
     *
     * @Route("/orders/{id}/return", name="orders_return")
     */
    public function return(Request $request, $id): Response
    {
        /** @var \Juweliere\UserBundle\Entity\User $user */
        $order = $this->juwApi->orders->get($id);

        $form = $this->createForm(ReturnAnnouncedFormType::class, $order);
        $form->handleRequest($request);

        $error = false;

        $template = '@JuweliereUser/pages/user/orders/return.html.twig';

        if ($form->isSubmitted() && $form->isValid()) {

            if ($order->getId() === $form->get('id')->getData()) {
                try {
                    $message = "Retoure angekündigt durch den Kunden";
                    $order->setLog($message);
                    $order->setStatus(Constants::ORDER_STATUS_RETURN_ANNOUNCED);
                    if ($this->juwApi->orders->update($order)) {
                        $template = '@JuweliereUser/pages/user/orders/return_announced.html.twig';
                    }
                } catch (\Throwable $e) {
                    $this->logger->error("ORDER RETURN ERROR: " .$e->getMessage());
                }

                $error = true;
            }
        }

        $response = $this->render(
            $template,
            [
                'order' => $order,
                'returnForm' => $form->createView(),
                'error' => $error
            ]);

        return $response;
    }

    /**
     * Display & process form to request a password reset.
     *
     * @Route("/orders/{id}/cancel", name="orders_cancel")
     */
    public function cancel(Request $request, $id): Response
    {
        /** @var \Juweliere\UserBundle\Entity\User $user */
        $order = $this->juwApi->orders->get($id);

        $form = $this->createForm(CancelOrderFormType::class, $order);
        $form->handleRequest($request);

        $error = false;

        $template = '@JuweliereUser/pages/user/orders/cancel.html.twig';

        if ($form->isSubmitted() && $form->isValid()) {

            if ($order->getId() === $form->get('id')->getData()) {
                try {
                    $message = "Stornierungsankündigung durch den Kunden";
                    $order->setLog($message);
                    $order->setStatus(Constants::ORDER_STATUS_CANCELED);
                    if ($this->juwApi->orders->update($order)) {
                        $template = '@JuweliereUser/pages/user/orders/canceled.html.twig';
                    }
                } catch (\Throwable $e) {
                    $this->logger->error("ORDER CANCEL ERROR: " .$e->getMessage());
                }

                $error = true;
            }
        }

        $response = $this->render(
            $template,
            [
                'order' => $order,
                'cancelForm' => $form->createView(),
                'error' => $error
            ]);

        return $response;
    }

    private function mapStatusLabel($status)
    {
        $mapping = array(
            "INIT" => "Eingegangen",
            "NEW" => "Eingegangen",
            "SHIPPED" => "Versandt",
            "CANCELED" => "Stornierung angekündigt",
            "REFUNDED" => "Storniert",
            "REFUNDED_NO_TRANSACTION" => "Storniert",
            "RETURNED" => "Retoure eingegangen",
            "RETURN_ANNOUNCED" => "Retoure angekündigt",
            "DONE" => "Abgeschlossen",
        );

        if (isset($mapping[$status])) {
            return $mapping[$status];
        }

        return "In Bearbeitung";
    }
}
