<?php
declare(strict_types=1);

namespace Juweliere\UserBundle\Controller;

use Juweliere\JuwApiClient\Entity\Address;
use Juweliere\JuwApiClient\Entity\User;
use GuzzleHttp\Exception\ClientException;
use Juweliere\CheckoutBundle\Constants;
use Juweliere\UserBundle\Form\AddressFormType;
use Juweliere\UserBundle\Form\CancelOrderFormType;
use Juweliere\UserBundle\Form\ReturnAnnouncedFormType;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route(path: '/user', name: 'juweliere_user_user_')]
class OrdersController extends BaseController
{

    /**
     * Display & process form to request a password reset.
     */
    #[Route(path: '/orders', name: 'orders')]
    public function list() : Response
    {
        $template = '@JuweliereUser/pages/user/orders.html.twig';
        $displayOrders = [];
        $hideOrderStatus = ["NEW", "INIT"];
        /** @var User $orderUser */
        try {
            $orders = $this->juwApi->orders->getByUser($this->getUser()->getId())->getOrders();
            foreach ($orders as $order) {
                if (!in_array($order->getStatus(), $hideOrderStatus)) {
                    $order->setStatus($this->mapStatusLabel($order->getStatus()));
                    $displayOrders[] = $order;
                }
            }
        } catch (ClientException) {

        }
        return $this->render(
            $template,
            [
                'orders' => $displayOrders,
                'user' => $this->getUser()
            ]);
    }

    /**
     * Display & process form to request a password reset.
     */
    #[Route(path: '/orders/{id}', name: 'orders_detail')]
    public function new(Request $request, $id): Response
    {
        $newAddress=  new Address();
        $form = $this->createForm(AddressFormType::class, $newAddress);
        $form->handleRequest($request);

        $order = $this->juwApi->orders->get($id);
        $orderStatusCode = $order->getStatus();

        $order->setStatus($this->mapStatusLabel($order->getStatus()));

        $template = '@JuweliereUser/pages/user/orders/detail.html.twig';

        return $this->render(
            $template,
            [
                'orderStatusCode' => $orderStatusCode,
                'order' => $order,
                'user' => $this->getUser(),
                'countries' => $this->getCountries()
            ]);
    }

    /**
     * Display & process form to request a password reset.
     */
    #[Route(path: '/orders/{id}/return', name: 'orders_return')]
    public function return(Request $request, $id): Response
    {
        $order = $this->juwApi->orders->get($id);

        $form = $this->createForm(ReturnAnnouncedFormType::class, $order);
        $form->handleRequest($request);

        $error = false;

        $template = '@JuweliereUser/pages/user/orders/return.html.twig';

        if ($form->isSubmitted() && $form->isValid()) {

            if ($order->getId() === $form->get('id')->getData()) {
                try {
                    $message = "Retoure angekündigt durch den Kunden";
                    $order->setLog($message);
                    $order->setStatus(Constants::ORDER_STATUS_RETURN_ANNOUNCED);
                    if ($this->juwApi->orders->update($order)) {
                        $template = '@JuweliereUser/pages/user/orders/return_announced.html.twig';
                    }
                } catch (\Throwable $e) {
                    $this->logger->error("ORDER RETURN ERROR: " .$e->getMessage());
                }

                $error = true;
            }
        }

        return $this->render(
            $template,
            [
                'order' => $order,
                'returnForm' => $form->createView(),
                'error' => $error
            ]);
    }

    /**
     * Display & process form to request a password reset.
     */
    #[Route(path: '/orders/{id}/cancel', name: 'orders_cancel')]
    public function cancel(Request $request, $id): Response
    {
        $order = $this->juwApi->orders->get($id);

        $form = $this->createForm(CancelOrderFormType::class, $order);
        $form->handleRequest($request);

        $error = false;

        $template = '@JuweliereUser/pages/user/orders/cancel.html.twig';

        if ($form->isSubmitted() && $form->isValid()) {

            if ($order->getId() === $form->get('id')->getData()) {
                try {
                    $message = "Stornierungsankündigung durch den Kunden";
                    $order->setLog($message);
                    $order->setStatus(Constants::ORDER_STATUS_CANCELED);
                    if ($this->juwApi->orders->update($order)) {
                        $template = '@JuweliereUser/pages/user/orders/canceled.html.twig';
                    }
                } catch (\Throwable $e) {
                    $this->logger->error("ORDER CANCEL ERROR: " .$e->getMessage());
                }

                $error = true;
            }
        }

        return $this->render(
            $template,
            [
                'order' => $order,
                'cancelForm' => $form->createView(),
                'error' => $error
            ]);
    }

    private function mapStatusLabel($status): string
    {
        $mapping = ["INIT" => "Eingegangen", "NEW" => "Eingegangen", "SHIPPED" => "Versandt", "CANCELED" => "Stornierung angekündigt", "REFUNDED" => "Storniert", "REFUNDED_NO_TRANSACTION" => "Storniert", "RETURNED" => "Retoure eingegangen", "RETURN_ANNOUNCED" => "Retoure angekündigt", "DONE" => "Abgeschlossen"];

        return $mapping[$status] ?? "In Bearbeitung";
    }
}
