<?php
declare(strict_types=1);

namespace Juweliere\UserBundle\Controller;

use Juweliere\ApiBundle\Service\JuwApi;
use Juweliere\JuwApiClient\Entity\User\Recovery;
use Juweliere\UserBundle\Form\ChangePasswordFormType;
use Juweliere\UserBundle\Form\ResetPasswordRequestFormType;
use Juweliere\UserBundle\User\Mailer;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;

#[Route(path: '/user', name: 'juweliere_user_user_')]
class RecoveryController extends AbstractController
{
    const string SESSION_KEY_RECOVERY_TOKEN = "recoveryToken";
    /**
     * RecoveryController constructor.
     */
    public function __construct(private readonly JuwApi $juwApi)
    {
    }

    /**
     * Display & process form to request a password reset.
     */
    #[Route(path: '/recovery', name: 'recovery')]
    public function request(Request $request, Mailer $mailer): Response
    {
        $form = $this->createForm(ResetPasswordRequestFormType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            return $this->processSendingPasswordResetEmail(
                $form->get('email')->getData(),
                $mailer
            );
        }

        return $this->render('@JuweliereUser/pages/user/recovery/recovery.html.twig', [
            'requestForm' => $form->createView(),
        ]);
    }

    /**
     * Confirmation page after a user has requested a password reset.
     */
    #[Route(path: '/checkmail', name: 'recovery_check_email')]
    public function checkEmail() : Response
    {
        return $this->render('@JuweliereUser/pages/user/recovery/check_email.html.twig');
    }

    /**
     * Validates and process the reset URL that the user clicked in their email.
     */
    #[Route(path: '/reset/{token}', name: 'reset_token')]
    public function reset(Request $request, UserPasswordHasherInterface $passwordEncoder, ?string $token = null): Response
    {
        if ($token) {
            // We store the token in session and remove it from the URL, to avoid the URL being
            // loaded in a browser and potentially leaking the token to 3rd party JavaScript.
            $request->getSession()->set(self::SESSION_KEY_RECOVERY_TOKEN, $token);

            return $this->redirectToRoute('juweliere_user_user_reset_token');
        }

        $token = $request->getSession()->get(self::SESSION_KEY_RECOVERY_TOKEN);

        if (null === $token) {
            throw $this->createNotFoundException('No reset password token found in the URL or in the session.');
        }

        $form = $this->createForm(ChangePasswordFormType::class);
        $form->handleRequest($request);

        $errors = false;

        if ($form->isSubmitted() && $form->isValid()) {

            try {
                $user = $this->juwApi->users->recover($token, $form->get('password')->getData());
                $request->getSession()->remove(self::SESSION_KEY_RECOVERY_TOKEN);

                return $this->redirectToRoute('juweliere_user_user_login');
            } catch (\Throwable) {
                $errors = true;
            }
        }

        return $this->render('@JuweliereUser/pages/user/recovery/reset.html.twig', [
            'resetForm' => $form->createView(),
            'errors' => $errors
        ]);
    }

    private function processSendingPasswordResetEmail(string $username, Mailer $mailer): RedirectResponse
    {
        $recovery = new Recovery();
        $recovery->setUsername($username);

        try {
            $recoveryToken = $this->juwApi->users->createRecoveryToken($recovery);
        } catch (\Throwable) {
            return $this->redirectToRoute('juweliere_user_user_recovery_check_email');
        }

        $recoverTokenUrl = $this->generateUrl(
            "juweliere_user_user_reset_token",
            ["token" => $recoveryToken],
            UrlGeneratorInterface::ABSOLUTE_URL
        );

        $mailer->sendPasswordRecover($username, $recoverTokenUrl);

        return $this->redirectToRoute('juweliere_user_user_recovery_check_email');
    }
}
