<?php
declare(strict_types=1);

namespace Juweliere\UserBundle\Controller;

use Juweliere\JuwApiClient\Entity\User\Registration;
use Juweliere\UserBundle\Form\RegistrationFormType;
use Juweliere\UserBundle\Security\UserProvider;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Core\Authentication\Token\UsernamePasswordToken;

#[Route(path: '/user', name: 'juweliere_user_user_')]
class RegistrationController extends BaseController
{
    /**
     * Display & process form to request a password reset.
     */
    #[Route(path: '/register', name: 'register')]
    public function request(Request $request, UserProvider $userProvider): Response
    {
        $newUser = new Registration();

        $form = $this->createForm(RegistrationFormType::class, $newUser);
        $form->handleRequest($request);

        if (!empty($newUser->getEmailRepeat())) {
            $this->logger->error("Registration honeypot: " . json_encode($request->request->all()));
            return $this->redirectToRoute('juweliere_user_user_register_success');
        }

        $error = false;

        if ($form->isSubmitted() && $form->isValid()) {

            if (!$this->checkSt($request->request->get("st"), $newUser->getEmail())) {
                $this->logger->error("Registration honeypot (submit time): " . json_encode($request->request->all()));
                return $this->redirectToRoute('juweliere_user_user_register_success');
            }

            try {

                $userApi = $this->juwApi->users->create($newUser);

                $user = $userProvider->mapApiUser($userApi);

                $token = new UsernamePasswordToken($user, 'main', $user->getRoles());

                $this->container->get('security.token_storage')->setToken($token);
                $this->session->set('_security_main', serialize($token));
                $this->session->set('_new_user', true);

                return $this->redirectToRoute('juweliere_user_user_register_success');

            } catch (\Throwable) {
                $error = true;
            }

        }

        $template = '@JuweliereUser/pages/user/register.html.twig';

        return $this->render(
            $template,
            [
                'registrationForm' => $form->createView(),
                'error' => $error
            ]);
    }



    /**
     * Display & process form to request a password reset.
     */
    #[Route(path: '/register/success', name: 'register_success')]
    public function success() : Response
    {
        if ($redirect = $this->session->get(LoginController::SESSION_LOGIN_REDIRECT)) {
            return new RedirectResponse($redirect);
        }
        #return $this->redirectToRoute('juweliere_user_user_profile_edit');
        $template = '@JuweliereUser/pages/user/profile.html.twig';
        $newUser = $this->session->get('_new_user');
        if ($newUser) {
            $this->session->set('_new_user', false);

            return $this->render(
                $template,
                [
                    'user' => $this->getUser(),
                    'newUser' => $newUser
                ]);
        }
        return $this->redirectToRoute('juweliere_user_user_profile');
    }

    private function checkSt(string|int|float|bool|null $st, ?string $email): bool
    {
        if (!empty($st) && strlen($st) > 2) {
            $checkSt = substr((string) $email, 1, 1) . substr((string) $email, -2, 1);
            if ($checkSt === substr($st, 0, 2) && intval(substr($st, 2) > 3)) {
                return true;
            }
        }

        return false;
    }
}
