<?php

declare(strict_types=1);

namespace Juweliere\UserBundle\Controller;

use Juweliere\JuwApiClient\Entity\User\WishlistItem;
use Juweliere\UserBundle\Entity\JuweliereUser;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;

#[Route(path: '/user', name: 'juweliere_user_user_')]
class WishlistController extends BaseController
{

    /**
     * @throws TransportExceptionInterface
     * @throws ServerExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws ClientExceptionInterface
     */
    #[Route(path: '/wishlist', name: 'wishlist')]
    public function wishlist() : Response
    {
        $wishlistItems = $this->getUser()->getWishlistItems();
        $products = [];
        if ($wishlistItems) {
            /** @var WishlistItem $wishlistItem */
            foreach ($wishlistItems as $wishlistItem) {
                $product = $this->productDataProvider->getProductBySku($wishlistItem->getSku());
                if ($product) {
                    $products[] = $product;
                }

            }
        }
        return $this->render('@JuweliereUser/pages/user/wishlist.html.twig', [
            'products' => $products,
        ]);
    }

    /**
     * @throws TransportExceptionInterface
     * @throws ServerExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws ClientExceptionInterface
     */
    #[Route(path: '/wishlist/count', name: 'wishlist_count')]
    public function wishlistCount(Request $request): Response
    {
        if (is_null($request->getContentTypeFormat()) || !str_contains($request->getContentTypeFormat(), "json")) {
            return $this->redirectToRoute("juweliere_user_user_profile");
        }

        $wishlistItems = $this->getUser()->getWishlistItems();
        $count = 0;

        if ($wishlistItems) {
            foreach ($wishlistItems as $wishlistItem) {
                $product = $this->productDataProvider->getProductBySku($wishlistItem->getSku());
                if ($product) {
                    $count++;
                }
            }
        }

        return new JsonResponse(["count" => $count]);
    }

    #[Route(path: '/wishlist/items', name: 'wishlist_items')]
    public function items(Request $request): Response
    {

        if (is_null($request->getContentTypeFormat()) || !str_contains($request->getContentTypeFormat(), "json")) {
            return $this->redirectToRoute("juweliere_user_user_profile");
        }

        $wishlistItems = $this->getUser()->getWishlistItems();

        $products = [];
        $skus = [];

        if ($wishlistItems) {
            /** @var WishlistItem $wishlistItem */
            foreach ($wishlistItems as $wishlistItem) {
                $product = $this->productDataProvider->getProductBySku($wishlistItem->getSku());
                if ($product) {
                    $skus[] = $wishlistItem->getSku();
                    $products[] = $product;
                }

            }
        }

        $content = $this->renderView('@JuweliereUser/snippets/slide-menu/wishlistitems.html.twig', [
            'products' => $products,
        ]);

        return new JsonResponse(["content" => $content, "skus" => $skus]);
    }

    #[Route(path: '/wishlist/skus', name: 'wishlist_skus')]
    public function wishlistSkus(Request $request): Response
    {
        if (is_null($request->getContentTypeFormat()) || !str_contains($request->getContentTypeFormat(), "json")) {
            return $this->redirectToRoute("juweliere_user_user_profile");
        }

        $wishlistItems = $this->getUser()->getWishlistItems();
        $products = [];

        if ($wishlistItems) {
            /** @var WishlistItem $wishlistItem */
            foreach ($wishlistItems as $wishlistItem) {
                    $products[] = $wishlistItem->getSku();
            }
        }

        return new JsonResponse($products);
    }

    #[Route(path: '/wishlist/remove', name: 'wishlist_remove_xhr')]
    public function removeFromWishlistXhr(Request $request): Response
    {
        /** @var User $user */
        $user = $this->getUser();

        $data = json_decode($request->getContent(), true);

        $wishlistItem = new WishlistItem();
        $wishlistItem->setSku($data["sku"]);
        $wishlistItem->setUser($user->getId());

        try {
            $this->juwApi->wishlistItems->addItemToWishlist($wishlistItem);
            $this->sessionRefreshUser();
        } catch (\Throwable $throwable) {
            var_dump($throwable->getMessage());
            die();
        }

        return new JsonResponse(["result" => true]);
    }

    #[Route(path: '/wishlist/add', name: 'wishlist_add_xhr')]
    public function addToWishlistXhr(Request $request): Response
    {
        /** @var User $user */
        $user = $this->getUser();

        $data = json_decode($request->getContent(), true);

        $wishlistItem = new WishlistItem();
        $wishlistItem->setSku($data["sku"]);
        $wishlistItem->setUser($user->getId());

        try {
            $this->juwApi->wishlistItems->addItemToWishlist($wishlistItem);
            $this->sessionRefreshUser();
        } catch (\Throwable $throwable) {
            var_dump($throwable->getMessage());
            die();
        }

        return new JsonResponse(["result" => true]);
    }

    #[Route(path: '/wishlist/{sku}/add', name: 'wishlist_add')]
    public function addToWishlist(string $sku): Response
    {
        /** @var JuweliereUser $user */
        $user = $this->getUser();

        $wishlistItem = new WishlistItem();
        $wishlistItem->setSku($sku);
        $wishlistItem->setUser($user->getId());

        try {
            $this->juwApi->wishlistItems->addItemToWishlist($wishlistItem);
            $this->sessionRefreshUser();
        } catch (\Throwable $throwable) {
            var_dump($throwable->getMessage());
            die();
        }

        if ($redirect = $this->session->get(LoginController::SESSION_LOGIN_ORIGIN)) {
            $this->session->remove(LoginController::SESSION_LOGIN_ORIGIN);
            return new RedirectResponse($redirect);
        }

        return $this->redirectToRoute('juweliere_user_user_wishlist');
    }

    #[Route(path: '/wishlist/{sku}/remove', name: 'wishlist_remove')]
    public function removeFromWishlist(Request $request, string $sku): Response
    {
        try {
        $wishlistItems = $this->getUser()->getWishlistItems();

        if ($wishlistItems) {
            /** @var WishlistItem $wishlistItem */
            foreach ($wishlistItems as $wishlistItem) {
                if ($wishlistItem->getSku() == $sku) {
                    try {
                        $this->juwApi->wishlistItems->deleteItemFromWishlist($wishlistItem->getId());
                        $this->sessionRefreshUser();
                    } catch (\Throwable $e) {
                        var_dump($e->getMessage());
                        die();
                    }
                }
            }
        }
        } catch (\Throwable $throwable) {
            var_dump($throwable->getMessage());
        }

        if ($request->isXmlHttpRequest() || $request->getContentTypeFormat() === "json") {
            return new JsonResponse(["result" => true]);
        } else {
            return $this->redirectToRoute('juweliere_user_user_wishlist');
        }
    }
}
